<?php

namespace Drupal\phonepay_payment;

use Methods\CustomCheckout;
use Methods\RecurringCheckout;
use Methods\StandardCheckout;

/**
 * Phonepe class.
 */
class PhonePe {

  /**
   * Instance for Phonepe.
   *
   * @var \Drupal\phonepay_payment
   */
  private static $instance;

  /**
   * Config service.
   *
   * @var \Drupal\Core\Config\Config
   */
  private $config;

  private function __construct($config) {
    $this->config = $config;
  }

  /**
   * Instance of Phonepe class using init() method  .
   *
   * @return PhonePe
   *   Instance of the Phonepe.
   */
  public static function getInstance() {
    if (NULL === static::$instance) {
      throw new \Exception("PhonePe not initialized");
    }

    return static::$instance;
  }

  /**
   * Initialize the PhonePe class.
   *
   * @param string $merchantId
   *   Payment merchant id.
   * @param string $merchantUserId
   *   Payment merchant user id.
   * @param string $salt_key
   *   Salt key for payment.
   * @param string $salt_index
   *   Salt index for payment.
   * @param string $redirectUrl
   *   Can be defined on per transaction basis.
   * @param string $callbackUrl
   *   Can be defined on per transaction basis.
   * @param string $mode
   *   DEV or PROD.
   * @param string $redirectMode
   *   Redirect or post.
   */
  public static function init($merchantId, $merchantUserId, $salt_key, $salt_index, $redirectUrl = "", $callbackUrl = "", $mode = "DEV", $redirectMode = "redirect") {
    if (NULL === static::$instance) {
      if ($mode == "PROD") {
        $host = "https://api.phonepe.com/apis/hermes";
      }
      else {
        $host = "https://api-preprod.phonepe.com/apis/pg-sandbox";
      }

      static::$instance = new static(
            [
              "MERCHANT_ID" => $merchantId,
              "MERCHANT_USER_ID" => $merchantUserId,
              "SALT_KEY" => $salt_key,
              "SALT_INDEX" => $salt_index,
              "REDIRECT_URL" => $redirectUrl,
              "CALLBACK_URL" => $callbackUrl,
              "HOST" => $host,
              "REDIRECT_MODE" => $redirectMode,
            ]
        );
    }

    return static::$instance;
  }

  /**
   * Get the config of the PhonePe class.
   *
   * @param string $key
   *   Config keys.
   *
   * @return array
   *   A array of config keys.
   */
  public function getConfig($key = NULL) {
    if ($key) {
      return $this->config[$key];
    }

    return $this->config;
  }

  /**
   * Get the standard checkout method.
   *
   * @return \Methods\StandardCheckout
   *   A method for standard checkout.
   */
  public function standardCheckout() {
    return new StandardCheckout();
  }

  /**
   * Get the custom checkout method.
   *
   * @return \Methods\CustomCheckout
   *   A method for custom checkout.
   */
  public function customCheckout() {
    return new CustomCheckout();
  }

  /**
   * Get the recurring checkout method.
   *
   * @return \Methods\RecurringCheckout
   *   A method for recurring checkout.
   */
  public function recurringCheckout() {
    return new RecurringCheckout();
  }

}
