<?php

namespace Drupal\phonepay_payment;

use GuzzleHttp\Client;

/**
 * Phonepe base class.
 */
class PhonePeBaseClass {

  /**
   * Config service.
   *
   * @var \Drupal\phonepay_payment
   */
  protected $config;

  /**
   * Client service.
   *
   * @var \Drupal\phonepay_payment
   */
  protected $client;

  public function __construct() {
    $this->client = new Client();
    $this->config = PhonePe::getInstance()->getConfig();
  }

  /**
   * Calculate the X-VERIFY header using the phonepe's algorithm.
   *
   * @param string $payload
   *   Payload string of payment.
   * @param string $path
   *   Path sting for payment.
   * @param string $saltKey
   *   Saltkey for payment.
   * @param string $saltIndex
   *   SaltIndex for payment.
   *
   * @return string
   *   Return Payment is vertified wherethere is it completed or not.
   */
  protected function calculateHeaders($payload, $path, $saltKey, $saltIndex) {
    $xVerify = hash('sha256', $payload . $path . $saltKey) . "###" . $saltIndex;

    return $xVerify;
  }

  /**
   * Executes the request and returns the response.
   *
   * @param string $method
   *   Payment method.
   * @param string $path
   *   Payment path.
   * @param array|null $payload
   *   Payment payload.
   * @param array $headers
   *   Payment headers.
   *
   * @return array
   *   return json with the payment execution request.
   */
  protected function executeRequest($method, $path, $payload = NULL, $headers = []) {
    if ($payload) {
      // Making Original Payload.
      $base64Payload = base64_encode(json_encode($payload));
      $requestBody = json_encode(['request' => $base64Payload]);

      // Calculate X-VERIFY.
      $xVerify = $this->calculateHeaders($base64Payload, $path, $this->config['SALT_KEY'], $this->config['SALT_INDEX']);
    }
    else {
      $requestBody = NULL;
      $xVerify = $this->calculateHeaders("", $path, $this->config['SALT_KEY'], $this->config['SALT_INDEX']);
    }

    // Set the headers for PhonePe's Algorithm.
    $headers['Content-Type'] = 'application/json';
    $headers['X-VERIFY'] = $xVerify;

    // Define the options.
    $options = [
      'headers' => $headers,
      'body' => $requestBody,
    ];

    $response = $this->client->request($method, $this->config['HOST'] . $path, $options);
    return json_decode($response->getBody(), TRUE);
  }

  /**
   * Verifies the response from the server using phonepe's algorithm.
   *
   * @param string $payload
   *   Payment payload.
   * @param string $xVerify
   *   Payment verification.
   *
   * @return bool
   *   return True if payment is verified.
   */
  protected function verifyResponse($payload, $xVerify) {
    $xVerify = explode("###", $xVerify);
    $xVerify = $xVerify[0];
    $xVerify = $xVerify . "###" . $this->config['SALT_INDEX'];

    $xVerifyCalculated = $this->calculateHeaders($payload, "", $this->config['SALT_KEY'], $this->config['SALT_INDEX']);

    if ($xVerify == $xVerifyCalculated) {
      return TRUE;
    }

    return FALSE;
  }

}
