# PhotoPrism Integration

**A comprehensive service-oriented integration with PhotoPrism, the AI-powered photo management solution.**

## Overview

The PhotoPrism Integration module provides a robust **Service-Oriented Architecture (SOA)** foundation for integrating [PhotoPrism](https://photoprism.app/) with Drupal. This module focuses on offering a comprehensive service layer that developers can use to build custom PhotoPrism integrations tailored to their specific needs.

## What is PhotoPrism?

PhotoPrism is an AI-powered photo management application that provides features like:
- Automatic photo classification using TensorFlow
- Face recognition and people detection
- Geographic mapping and location-based organization
- Advanced search with natural language support
- Album management and timeline views
- WebDAV support for syncing

## Service-Oriented Architecture

This module is built on **Service-Oriented Architecture** principles, providing a clean, injectable `PhotoPrismClient` service that encapsulates all communication with the PhotoPrism API. Developers can inject this service into their custom modules, blocks, controllers, or plugins.

### Why Service-Oriented?

- **Maximum Flexibility**: Build exactly the features you need
- **Clean Code**: Reusable service methods following Drupal best practices
- **Easy Extension**: No need to modify the base module
- **Minimal Dependencies**: Lightweight foundation with no UI opinions
- **Developer-Friendly**: Comprehensive API coverage with extensive documentation

## Installation

1. Copy the `photoprism_integration` folder to `/modules/custom/`
2. Enable the module via Drupal admin or: `drush en photoprism_integration`
3. Configure your PhotoPrism server connection at `/admin/config/media/photoprism`

## Configuration

1. Navigate to **Configuration → Media → PhotoPrism Integration**
2. Enter your PhotoPrism server URL (e.g., `http://localhost:2342`)
3. Enter your access token or app password
4. Click "Test Connection" to verify
5. Save configuration

### Generating Access Tokens

**Via PhotoPrism CLI:**
```bash
# Generate app password for a user
docker exec photoprism photoprism auth add -n "MyApp" -s "*" admin

# Generate access token (no user account)
docker exec photoprism photoprism auth add
```

**Via PhotoPrism Web UI:**
1. Navigate to Settings → Account → Apps and Devices
2. Generate a new app password

## Available Service Methods

### Server Operations

```php
$photoprism = \Drupal::service('photoprism_integration.client');

// Test connection
$photoprism->testConnection();

// Get server status
$photoprism->getStatus();

// Get server configuration
$photoprism->getConfig();

// Get preview token for thumbnails
$photoprism->getPreviewToken();
```

### Album Operations

```php
// List all albums
$photoprism->getAlbums();

// Get specific album
$photoprism->getAlbum($album_uid);

// Create new album
$photoprism->createAlbum('Vacation 2024', 'Summer trip photos');

// Update album
$photoprism->updateAlbum($album_uid, ['Title' => 'New Name']);

// Delete album
$photoprism->deleteAlbum($album_uid);

// Get photos in album
$photoprism->getAlbumPhotos($album_uid);

// Add photos to album
$photoprism->addPhotosToAlbum($album_uid, [$photo_uid1, $photo_uid2]);

// Remove photos from album
$photoprism->removePhotosFromAlbum($album_uid, [$photo_uid1]);
```

### Photo Operations

```php
// Get all photos
$photoprism->getPhotos();

// Get photos with filters
$photoprism->getPhotos(['count' => 50, 'offset' => 0]);

// Get specific photo
$photoprism->getPhoto($photo_uid);

// Update photo
$photoprism->updatePhoto($photo_uid, ['Title' => 'Beach Sunset', 'Favorite' => true]);

// Delete photos
$photoprism->deletePhotos([$photo_uid1, $photo_uid2]);

// Like/unlike photo
$photoprism->likePhoto($photo_uid);
$photoprism->unlikePhoto($photo_uid);

// Get thumbnail URL
$photoprism->getThumbnailUrl($hash, 'tile_224');

// Get download URL
$photoprism->getDownloadUrl($hash);
```

### Search Operations

```php
// Search photos by query
$photoprism->searchPhotos('sunset beach');

// Search with parameters
$photoprism->searchPhotos('vacation', ['count' => 20]);
```

### Labels (Tags)

```php
// Get all labels
$photoprism->getLabels();

// Update label
$photoprism->updateLabel($label_uid, ['Name' => 'Nature']);
```

### People (Face Recognition)

```php
// Get all recognized people
$photoprism->getPeople();

// Get specific person
$photoprism->getPerson($person_uid);

// Update person name
$photoprism->updatePerson($person_uid, ['Name' => 'John Doe']);
```

### Places (Locations)

```php
// Get all places
$photoprism->getPlaces();
```

### Folders

```php
// Get all folders
$photoprism->getFolders();
```

### Moments (Auto-generated Albums)

```php
// Get moments
$photoprism->getMoments();
```

### Calendar

```php
// Get calendar (photos grouped by month)
$photoprism->getCalendar();
```

## Example: Custom Block

```php
<?php

namespace Drupal\my_module\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\photoprism_integration\Service\PhotoPrismClient;

/**
 * @Block(
 *   id = "photoprism_recent_photos",
 *   admin_label = @Translation("PhotoPrism Recent Photos"),
 * )
 */
class RecentPhotosBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected $photoPrismClient;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, PhotoPrismClient $photoprism_client) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->photoPrismClient = $photoprism_client;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('photoprism_integration.client')
    );
  }

  public function build() {
    $photos = $this->photoPrismClient->getPhotos(['count' => 10]);

    $items = [];
    foreach ($photos as $photo) {
      $hash = $photo['Files'][0]['Hash'] ?? NULL;
      if ($hash) {
        $items[] = [
          '#markup' => '<img src="' . $this->photoPrismClient->getThumbnailUrl($hash) . '" />',
        ];
      }
    }

    return ['#theme' => 'item_list', '#items' => $items];
  }
}
```

## Included Demo Pages

- **Settings**: `/admin/config/media/photoprism` - Configure connection
- **Browse Albums**: `/admin/config/media/photoprism/albums` - View your albums
- **Album View**: `/admin/config/media/photoprism/album/{uid}` - View photos in an album

## Error Handling

All service methods return `NULL` on failure and log errors to the `photoprism_integration` watchdog channel:

```php
$albums = $photoprism->getAlbums();
if ($albums === NULL) {
  // Check logs at /admin/reports/dblog
  \Drupal::messenger()->addError('Failed to fetch albums');
  return;
}
```

## Authentication

The module uses PhotoPrism's Bearer token authentication. The access token is passed in the `Authorization` header:

```
Authorization: Bearer <token>
```

## Thumbnail Sizes

Available thumbnail sizes for `getThumbnailUrl()`:

- `tile_50` - 50x50
- `tile_100` - 100x100
- `tile_224` - 224x224 (default)
- `tile_500` - 500x500
- `fit_720` - Fit within 720px
- `fit_1280` - Fit within 1280px
- `fit_1920` - Fit within 1920px
- `fit_2560` - Fit within 2560px

## Docker Networking Note

When running Drupal and PhotoPrism in Docker containers:
- Configure the server URL as `http://host.docker.internal:<port>` for inter-container communication
- Thumbnail URLs will contain this internal hostname, which browsers cannot resolve
- Consider implementing an image proxy for production use

## Requirements

- Drupal 9.4, 10, or 11
- PHP 7.4 or higher
- Access to a PhotoPrism server
- PhotoPrism access token or app password

## Extension Ideas

1. **Media Library Integration**: Import PhotoPrism photos as Drupal Media entities
2. **Views Integration**: Create Views plugins for PhotoPrism data
3. **Field Formatters**: Display PhotoPrism images in content types
4. **Caching Layer**: Add Drupal cache integration
5. **Image Proxy**: Proxy images through Drupal for browser compatibility
6. **Face Recognition UI**: Build interfaces using people detection data
7. **Map Integration**: Display photos on geographic maps

## License

GPL-2.0-or-later

## Credits

- PhotoPrism: https://photoprism.app/
- PhotoPrism Documentation: https://docs.photoprism.app/
