<?php

namespace Drupal\photoprism_integration\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Service for communicating with PhotoPrism API.
 *
 * This service provides a comprehensive interface to the PhotoPrism API,
 * allowing developers to build custom integrations and features.
 * All methods return NULL on failure and log errors for debugging.
 */
class PhotoPrismClient
{

    /**
     * The HTTP client.
     *
     * @var \GuzzleHttp\ClientInterface
     */
    protected $httpClient;

    /**
     * The config factory.
     *
     * @var \Drupal\Core\Config\ConfigFactoryInterface
     */
    protected $configFactory;

    /**
     * The logger.
     *
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * Cached preview token for thumbnail access.
     *
     * @var string|null
     */
    protected $previewToken;

    /**
     * Constructs a PhotoPrismClient object.
     *
     * @param \GuzzleHttp\ClientInterface $http_client
     *   The HTTP client.
     * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
     *   The config factory.
     * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
     *   The logger factory.
     */
    public function __construct(ClientInterface $http_client, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory)
    {
        $this->httpClient = $http_client;
        $this->configFactory = $config_factory;
        $this->logger = $logger_factory->get('photoprism_integration');
    }

    /**
     * Get the PhotoPrism server URL from configuration.
     *
     * @return string
     *   The server URL.
     */
    public function getServerUrl()
    {
        $config = $this->configFactory->get('photoprism_integration.settings');
        return rtrim($config->get('server_url'), '/');
    }

    /**
     * Get the access token from configuration.
     *
     * @return string
     *   The access token.
     */
    protected function getAccessToken()
    {
        $config = $this->configFactory->get('photoprism_integration.settings');
        return $config->get('access_token');
    }

    /**
     * Make a request to the PhotoPrism API.
     *
     * @param string $method
     *   The HTTP method (GET, POST, PUT, DELETE, etc.).
     * @param string $endpoint
     *   The API endpoint (e.g., '/api/v1/albums').
     * @param array $options
     *   Additional request options (body, query params, etc.).
     *
     * @return array|null
     *   The decoded JSON response or NULL on failure.
     */
    protected function request($method, $endpoint, array $options = [])
    {
        $server_url = $this->getServerUrl();
        $access_token = $this->getAccessToken();

        if (empty($server_url) || empty($access_token)) {
            $this->logger->error('PhotoPrism server URL or access token not configured.');
            return NULL;
        }

        $url = $server_url . $endpoint;

        // Add authentication header
        $options['headers']['Authorization'] = 'Bearer ' . $access_token;
        $options['headers']['Content-Type'] = 'application/json';
        $options['headers']['Accept'] = 'application/json';

        try {
            $response = $this->httpClient->request($method, $url, $options);
            $body = $response->getBody()->getContents();
            return json_decode($body, TRUE);
        } catch (GuzzleException $e) {
            $this->logger->error('PhotoPrism API request failed: @message', [
                '@message' => $e->getMessage(),
            ]);
            return NULL;
        }
    }

    // =========================================================================
    // SERVER ENDPOINTS
    // =========================================================================

    /**
     * Test the connection to the PhotoPrism server.
     *
     * @return array
     *   Array with 'success' boolean and 'message' string.
     */
    public function testConnection()
    {
        $result = $this->request('GET', '/api/v1/status');

        if ($result !== NULL) {
            return [
                'success' => TRUE,
                'message' => 'Successfully connected to PhotoPrism server.',
                'data' => $result,
            ];
        }

        return [
            'success' => FALSE,
            'message' => 'Failed to connect to PhotoPrism server. Please check your settings.',
        ];
    }

    /**
     * Get server status.
     *
     * @return array|null
     *   Server status or NULL on failure.
     */
    public function getStatus()
    {
        return $this->request('GET', '/api/v1/status');
    }

    /**
     * Get server configuration.
     *
     * @return array|null
     *   Server config or NULL on failure.
     */
    public function getConfig()
    {
        return $this->request('GET', '/api/v1/config');
    }

    /**
     * Get the preview token for thumbnail access.
     *
     * @return string|null
     *   The preview token or NULL on failure.
     */
    public function getPreviewToken()
    {
        if ($this->previewToken !== NULL) {
            return $this->previewToken;
        }

        $config = $this->getConfig();
        if ($config && isset($config['previewToken'])) {
            $this->previewToken = $config['previewToken'];
            return $this->previewToken;
        }

        return NULL;
    }

    // =========================================================================
    // ALBUM ENDPOINTS
    // =========================================================================

    /**
     * Get all albums.
     *
     * @param array $params
     *   Optional query parameters (count, offset, q, type, etc.).
     *
     * @return array|null
     *   Array of albums or NULL on failure.
     */
    public function getAlbums(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/albums', [
            'query' => $params,
        ]);
    }

    /**
     * Get a specific album by UID.
     *
     * @param string $album_uid
     *   The album UID.
     *
     * @return array|null
     *   Album data or NULL on failure.
     */
    public function getAlbum($album_uid)
    {
        return $this->request('GET', '/api/v1/albums/' . $album_uid);
    }

    /**
     * Create a new album.
     *
     * @param string $title
     *   The album title.
     * @param string|null $description
     *   Optional album description.
     *
     * @return array|null
     *   Created album data or NULL on failure.
     */
    public function createAlbum($title, $description = NULL)
    {
        $data = ['Title' => $title];
        if ($description) {
            $data['Description'] = $description;
        }

        return $this->request('POST', '/api/v1/albums', [
            'json' => $data,
        ]);
    }

    /**
     * Update an album.
     *
     * @param string $album_uid
     *   The album UID.
     * @param array $data
     *   Album data to update (Title, Description, etc.).
     *
     * @return array|null
     *   Updated album data or NULL on failure.
     */
    public function updateAlbum($album_uid, array $data)
    {
        return $this->request('PUT', '/api/v1/albums/' . $album_uid, [
            'json' => $data,
        ]);
    }

    /**
     * Delete an album.
     *
     * @param string $album_uid
     *   The album UID.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function deleteAlbum($album_uid)
    {
        return $this->request('DELETE', '/api/v1/albums/' . $album_uid);
    }

    /**
     * Get photos in an album.
     *
     * @param string $album_uid
     *   The album UID.
     * @param array $params
     *   Optional query parameters (count, offset, etc.).
     *
     * @return array|null
     *   Array of photos or NULL on failure.
     */
    public function getAlbumPhotos($album_uid, array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);
        $params['album'] = $album_uid;

        return $this->request('GET', '/api/v1/photos', [
            'query' => $params,
        ]);
    }

    /**
     * Add photos to an album.
     *
     * @param string $album_uid
     *   The album UID.
     * @param array $photo_uids
     *   Array of photo UIDs to add.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function addPhotosToAlbum($album_uid, array $photo_uids)
    {
        return $this->request('POST', '/api/v1/albums/' . $album_uid . '/photos', [
            'json' => ['photos' => $photo_uids],
        ]);
    }

    /**
     * Remove photos from an album.
     *
     * @param string $album_uid
     *   The album UID.
     * @param array $photo_uids
     *   Array of photo UIDs to remove.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function removePhotosFromAlbum($album_uid, array $photo_uids)
    {
        return $this->request('DELETE', '/api/v1/albums/' . $album_uid . '/photos', [
            'json' => ['photos' => $photo_uids],
        ]);
    }

    // =========================================================================
    // PHOTO ENDPOINTS
    // =========================================================================

    /**
     * Get all photos with optional filtering.
     *
     * @param array $params
     *   Query parameters for filtering (q, count, offset, etc.).
     *
     * @return array|null
     *   Array of photos or NULL on failure.
     */
    public function getPhotos(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/photos', [
            'query' => $params,
        ]);
    }

    /**
     * Get a specific photo by UID.
     *
     * @param string $photo_uid
     *   The photo UID.
     *
     * @return array|null
     *   Photo data or NULL on failure.
     */
    public function getPhoto($photo_uid)
    {
        return $this->request('GET', '/api/v1/photos/' . $photo_uid);
    }

    /**
     * Update a photo.
     *
     * @param string $photo_uid
     *   The photo UID.
     * @param array $data
     *   Photo data to update (Title, Description, Favorite, etc.).
     *
     * @return array|null
     *   Updated photo data or NULL on failure.
     */
    public function updatePhoto($photo_uid, array $data)
    {
        return $this->request('PUT', '/api/v1/photos/' . $photo_uid, [
            'json' => $data,
        ]);
    }

    /**
     * Delete photos.
     *
     * @param array $photo_uids
     *   Array of photo UIDs to delete.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function deletePhotos(array $photo_uids)
    {
        return $this->request('POST', '/api/v1/batch/photos/delete', [
            'json' => ['photos' => $photo_uids],
        ]);
    }

    /**
     * Like/favorite a photo.
     *
     * @param string $photo_uid
     *   The photo UID.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function likePhoto($photo_uid)
    {
        return $this->request('POST', '/api/v1/photos/' . $photo_uid . '/like');
    }

    /**
     * Unlike/unfavorite a photo.
     *
     * @param string $photo_uid
     *   The photo UID.
     *
     * @return array|null
     *   Response or NULL on failure.
     */
    public function unlikePhoto($photo_uid)
    {
        return $this->request('DELETE', '/api/v1/photos/' . $photo_uid . '/like');
    }

    /**
     * Get photo thumbnail URL.
     *
     * @param string $hash
     *   The photo/file hash.
     * @param string $size
     *   Thumbnail size (tile_50, tile_100, tile_224, tile_500, fit_720, fit_1280, fit_1920, fit_2560).
     *
     * @return string|null
     *   The thumbnail URL or NULL if preview token unavailable.
     */
    public function getThumbnailUrl($hash, $size = 'tile_224')
    {
        $server_url = $this->getServerUrl();
        $preview_token = $this->getPreviewToken();

        if (empty($preview_token)) {
            return NULL;
        }

        return $server_url . '/api/v1/t/' . $hash . '/' . $preview_token . '/' . $size;
    }

    /**
     * Get photo download URL.
     *
     * @param string $hash
     *   The photo/file hash.
     *
     * @return string
     *   The download URL.
     */
    public function getDownloadUrl($hash)
    {
        $server_url = $this->getServerUrl();
        return $server_url . '/api/v1/dl/' . $hash . '?t=' . $this->getAccessToken();
    }

    // =========================================================================
    // SEARCH ENDPOINTS
    // =========================================================================

    /**
     * Search photos.
     *
     * @param string $query
     *   Search query string.
     * @param array $params
     *   Additional search parameters (count, offset, etc.).
     *
     * @return array|null
     *   Search results or NULL on failure.
     */
    public function searchPhotos($query, array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);
        $params['q'] = $query;

        return $this->request('GET', '/api/v1/photos', [
            'query' => $params,
        ]);
    }

    // =========================================================================
    // LABEL ENDPOINTS
    // =========================================================================

    /**
     * Get all labels.
     *
     * @param array $params
     *   Optional query parameters (count, offset, q, etc.).
     *
     * @return array|null
     *   Array of labels or NULL on failure.
     */
    public function getLabels(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/labels', [
            'query' => $params,
        ]);
    }

    /**
     * Update a label.
     *
     * @param string $label_uid
     *   The label UID.
     * @param array $data
     *   Label data to update.
     *
     * @return array|null
     *   Updated label data or NULL on failure.
     */
    public function updateLabel($label_uid, array $data)
    {
        return $this->request('PUT', '/api/v1/labels/' . $label_uid, [
            'json' => $data,
        ]);
    }

    // =========================================================================
    // PEOPLE/FACES ENDPOINTS
    // =========================================================================

    /**
     * Get all people (faces).
     *
     * @param array $params
     *   Optional query parameters (count, offset, etc.).
     *
     * @return array|null
     *   Array of people or NULL on failure.
     */
    public function getPeople(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/subjects', [
            'query' => $params,
        ]);
    }

    /**
     * Get a specific person.
     *
     * @param string $person_uid
     *   The person UID.
     *
     * @return array|null
     *   Person data or NULL on failure.
     */
    public function getPerson($person_uid)
    {
        return $this->request('GET', '/api/v1/subjects/' . $person_uid);
    }

    /**
     * Update a person.
     *
     * @param string $person_uid
     *   The person UID.
     * @param array $data
     *   Person data to update (Name, etc.).
     *
     * @return array|null
     *   Updated person data or NULL on failure.
     */
    public function updatePerson($person_uid, array $data)
    {
        return $this->request('PUT', '/api/v1/subjects/' . $person_uid, [
            'json' => $data,
        ]);
    }

    // =========================================================================
    // PLACES ENDPOINTS
    // =========================================================================

    /**
     * Get places (locations).
     *
     * @param array $params
     *   Optional query parameters (count, offset, q, etc.).
     *
     * @return array|null
     *   Array of places or NULL on failure.
     */
    public function getPlaces(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/geo', [
            'query' => $params,
        ]);
    }

    // =========================================================================
    // FOLDER ENDPOINTS
    // =========================================================================

    /**
     * Get all folders.
     *
     * @param array $params
     *   Optional query parameters.
     *
     * @return array|null
     *   Array of folders or NULL on failure.
     */
    public function getFolders(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/folders/originals', [
            'query' => $params,
        ]);
    }

    // =========================================================================
    // FILE ENDPOINTS
    // =========================================================================

    /**
     * Get files for a photo.
     *
     * @param string $photo_uid
     *   The photo UID.
     *
     * @return array|null
     *   Array of files or NULL on failure.
     */
    public function getPhotoFiles($photo_uid)
    {
        $photo = $this->getPhoto($photo_uid);
        return $photo['Files'] ?? NULL;
    }

    // =========================================================================
    // MOMENTS ENDPOINTS
    // =========================================================================

    /**
     * Get moments (auto-generated albums based on time/location).
     *
     * @param array $params
     *   Optional query parameters.
     *
     * @return array|null
     *   Array of moments or NULL on failure.
     */
    public function getMoments(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/moments', [
            'query' => $params,
        ]);
    }

    // =========================================================================
    // CALENDAR ENDPOINTS
    // =========================================================================

    /**
     * Get calendar months (photos grouped by month).
     *
     * @param array $params
     *   Optional query parameters.
     *
     * @return array|null
     *   Array of calendar months or NULL on failure.
     */
    public function getCalendar(array $params = [])
    {
        $defaults = ['count' => 100, 'offset' => 0];
        $params = array_merge($defaults, $params);

        return $this->request('GET', '/api/v1/calendar', [
            'query' => $params,
        ]);
    }

}
