<?php

namespace Drupal\photoprism_integration\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\photoprism_integration\Service\PhotoPrismClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying PhotoPrism albums and photos.
 */
class PhotoPrismAlbumsController extends ControllerBase {

    /**
     * The PhotoPrism client service.
     *
     * @var \Drupal\photoprism_integration\Service\PhotoPrismClient
     */
    protected $photoPrismClient;

    /**
     * Constructs a PhotoPrismAlbumsController object.
     *
     * @param \Drupal\photoprism_integration\Service\PhotoPrismClient $photoprism_client
     *   The PhotoPrism client service.
     */
    public function __construct(PhotoPrismClient $photoprism_client) {
        $this->photoPrismClient = $photoprism_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container) {
        return new static(
            $container->get('photoprism_integration.client')
        );
    }

    /**
     * List all albums.
     *
     * @return array
     *   Render array.
     */
    public function listAlbums() {
        $albums = $this->photoPrismClient->getAlbums(['count' => 100]);

        if ($albums === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch albums from PhotoPrism. Please check your configuration.'));
            return [
                '#markup' => $this->t('Unable to connect to PhotoPrism server.'),
            ];
        }

        if (empty($albums)) {
            return [
                '#markup' => $this->t('No albums found in PhotoPrism.'),
            ];
        }

        $header = [
            $this->t('Title'),
            $this->t('Description'),
            $this->t('Photo Count'),
            $this->t('Type'),
            $this->t('Created'),
        ];

        $rows = [];
        foreach ($albums as $album) {
            $album_uid = $album['UID'] ?? '';
            $title = $album['Title'] ?? 'Untitled';

            // Calculate real photo count since API often returns 0
            $photo_count = 0;
            $params = ['count' => 500]; // Limit to avoid performance issues

            // Use 'Filter' if available (for folders), otherwise use 'album' ID
            if (!empty($album['Filter'])) {
                $params['q'] = $album['Filter'];
            } else {
                $params['album'] = $album_uid;
            }

            $album_photos = $this->photoPrismClient->getPhotos($params);

            if ($album_photos && !empty($album_photos)) {
                // Deduplicate by UID to handle sidecars
                $unique_photos = [];
                foreach ($album_photos as $photo) {
                    $p_uid = $photo['UID'] ?? '';
                    if ($p_uid) {
                        $unique_photos[$p_uid] = true;
                    }
                }
                $photo_count = count($unique_photos);
            }

            // Create a link to view the album
            $url = Url::fromRoute('photoprism_integration.album', ['album_uid' => $album_uid]);
            $link = Link::fromTextAndUrl($title, $url)->toString();

            $rows[] = [
                $link,
                $album['Description'] ?? '',
                $photo_count, // Use calculated count
                $album['Type'] ?? 'album',
                isset($album['CreatedAt']) ? date('Y-m-d', strtotime($album['CreatedAt'])) : '',
            ];
        }

        return [
            'table' => [
                '#type' => 'table',
                '#header' => $header,
                '#rows' => $rows,
                '#empty' => $this->t('No albums found.'),
            ],
            'help_text' => [
                '#type' => 'container',
                '#attributes' => ['class' => ['description', 'messages', 'messages--status']],
                '#markup' => $this->t('<strong>Note:</strong> "Folder" types represent physical directories on the server, while "Album" types are virtual collections. Duplicate entries (e.g., "January 2026") may appear if PhotoPrism automatically creates a folder view and a corresponding monthly album.'),
            ],
        ];
    }

    /**
     * View photos in an album.
     *
     * @param string $album_uid
     *   The album UID.
     *
     * @return array
     *   Render array.
     */
    public function viewAlbum($album_uid) {
        $album = $this->photoPrismClient->getAlbum($album_uid);

        if ($album === NULL) {
            $this->messenger()->addError($this->t('Failed to fetch album from PhotoPrism.'));
            return [
                '#markup' => $this->t('Unable to load album.'),
            ];
        }

        // Get photos in this album
        $raw_photos = $this->photoPrismClient->getAlbumPhotos($album_uid);

        if ($raw_photos === NULL || empty($raw_photos)) {
            return [
                '#markup' => $this->t('No photos found in this album.'),
            ];
        }

        // Deduplicate photos by UID (API returns sidecars as separate entries)
        $photos = [];
        foreach ($raw_photos as $photo) {
            $uid = $photo['UID'];
            if (!isset($photos[$uid])) {
                $photos[$uid] = $photo;
            }
        }

        $build = [
            '#type' => 'container',
            '#attributes' => ['class' => ['photoprism-album-gallery']],
        ];

        // Title is handled by _title_callback

        $build['count'] = [
            '#markup' => '<p>' . $this->t('@count photos', ['@count' => count($photos)]) . '</p>',
        ];

        $build['gallery'] = [
            '#type' => 'container',
            '#attributes' => [
                'style' => 'display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 16px;',
            ],
        ];

        foreach ($photos as $uid => $photo) {
            // Get the file hash for thumbnail
            $hash = NULL;
            if (isset($photo['Files']) && !empty($photo['Files'])) {
                $hash = $photo['Files'][0]['Hash'] ?? NULL;
            } elseif (isset($photo['Hash'])) {
                $hash = $photo['Hash'];
            }

            if ($hash) {
                $thumbnail_url = $this->photoPrismClient->getThumbnailUrl($hash, 'tile_224');
            } else {
                $thumbnail_url = NULL;
            }

            $title = $photo['Title'] ?? $photo['FileName'] ?? 'Photo';

            $build['gallery']['photo_' . $uid] = [
                '#type' => 'container',
                '#attributes' => [
                    'style' => 'text-align: center; padding: 8px; background: #f5f5f5; border-radius: 4px;',
                ],
                'image' => [
                    '#markup' => $thumbnail_url
                        ? '<img src="' . $thumbnail_url . '" alt="' . htmlspecialchars($title) . '" style="max-width: 100%; height: auto; border-radius: 4px;" />'
                        : '<div style="width: 200px; height: 200px; background: #ddd; display: flex; align-items: center; justify-content: center;">No preview</div>',
                ],
                'title' => [
                    '#markup' => '<p style="margin-top: 8px; font-size: 12px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">' . htmlspecialchars($title) . '</p>',
                ],
            ];
        }

        return $build;
    }

    /**
     * Title callback for album view.
     *
     * @param string $album_uid
     *   The album UID.
     *
     * @return string
     *   The page title.
     */
    public function albumTitle($album_uid) {
        $album = $this->photoPrismClient->getAlbum($album_uid);
        return $album['Title'] ?? $this->t('Album');
    }

}
