<?php

namespace Drupal\photoprism_integration\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\photoprism_integration\Service\PhotoPrismClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure PhotoPrism Integration settings.
 */
class PhotoPrismSettingsForm extends ConfigFormBase {

    /**
     * The PhotoPrism client service.
     *
     * @var \Drupal\photoprism_integration\Service\PhotoPrismClient
     */
    protected $photoPrismClient;

    /**
     * Constructs a PhotoPrismSettingsForm object.
     *
     * @param \Drupal\photoprism_integration\Service\PhotoPrismClient $photoprism_client
     *   The PhotoPrism client service.
     */
    public function __construct(PhotoPrismClient $photoprism_client) {
        $this->photoPrismClient = $photoprism_client;
    }

    /**
     * {@inheritdoc}
     */
    public static function create(ContainerInterface $container) {
        return new static(
            $container->get('photoprism_integration.client')
        );
    }

    /**
     * {@inheritdoc}
     */
    protected function getEditableConfigNames() {
        return ['photoprism_integration.settings'];
    }

    /**
     * {@inheritdoc}
     */
    public function getFormId() {
        return 'photoprism_integration_settings';
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(array $form, FormStateInterface $form_state) {
        $config = $this->config('photoprism_integration.settings');

        // Place connection result at the top
        $form['connection_result'] = [
            '#type' => 'container',
            '#attributes' => ['id' => 'connection-test-result'],
            '#weight' => -100,
        ];

        $form['server_url'] = [
            '#type' => 'url',
            '#title' => $this->t('PhotoPrism Server URL'),
            '#description' => $this->t('The base URL of your PhotoPrism server (e.g., http://localhost:2342 or https://photos.example.com). Do not include a trailing slash.'),
            '#default_value' => $config->get('server_url'),
            '#required' => TRUE,
        ];

        $form['access_token'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Access Token / App Password'),
            '#description' => $this->t('Your PhotoPrism access token or app password. You can generate one via CLI using "photoprism auth add" or from the PhotoPrism web UI under Settings > Account > Apps and Devices.'),
            '#default_value' => $config->get('access_token'),
            '#required' => TRUE,
            '#attributes' => [
                'autocomplete' => 'off',
            ],
        ];

        $form['test_connection'] = [
            '#type' => 'button',
            '#value' => $this->t('Test Connection'),
            '#ajax' => [
                'callback' => '::testConnectionAjax',
                'wrapper' => 'connection-test-result',
                'progress' => [
                    'type' => 'throbber',
                    'message' => $this->t('Testing connection...'),
                ],
            ],
        ];

        return parent::buildForm($form, $form_state);
    }

    /**
     * AJAX callback for testing the connection.
     */
    public function testConnectionAjax(array &$form, FormStateInterface $form_state) {
        // Temporarily save the form values to test with them
        $temp_config = $this->config('photoprism_integration.settings');
        $server_url = $form_state->getValue('server_url');
        $access_token = $form_state->getValue('access_token');

        // Save temporarily
        $temp_config->set('server_url', $server_url);
        $temp_config->set('access_token', $access_token);
        $temp_config->save();

        $result = $this->photoPrismClient->testConnection();

        if ($result['success']) {
            $form['connection_result']['#markup'] = '<div class="messages messages--status">' . $result['message'] . '</div>';

            // Try to get server status
            $status = $this->photoPrismClient->getStatus();
            if ($status) {
                $status_info = isset($status['status']) ? $status['status'] : 'OK';
                $form['connection_result']['#markup'] .= '<div class="messages messages--status">Server Status: ' . $status_info . '</div>';
            }
        } else {
            $form['connection_result']['#markup'] = '<div class="messages messages--error">' . $result['message'] . '</div>';
        }

        return $form['connection_result'];
    }

    /**
     * {@inheritdoc}
     */
    public function submitForm(array &$form, FormStateInterface $form_state) {
        $this->config('photoprism_integration.settings')
            ->set('server_url', rtrim($form_state->getValue('server_url'), '/'))
            ->set('access_token', $form_state->getValue('access_token'))
            ->save();

        parent::submitForm($form, $form_state);
    }

}
