<?php

declare(strict_types=1);

namespace Drupal\photos\Drush\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\photos\PhotosUploadInterface;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Photos drush commands.
 */
final class PhotosCommands extends DrushCommands {

  use AutowireTrait;

  /**
   * Constructs a PhotosCommands object.
   */
  public function __construct(
    protected configFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileSystemInterface $fileSystem,
    protected ImageFactory $imageFactory,
    #[Autowire(service: 'photos.upload')]
    protected PhotosUploadInterface $photosUpload,
  ) {
    parent::__construct();
  }

  /**
   * Import image files into existing album.
   *
   * Optionally create a new album.
   */
  #[CLI\Command(name: 'photos:import')]
  #[CLI\Argument(name: 'directory', description: 'Source directory of image files to import.')]
  #[CLI\Option(name: 'nid', description: 'Album node ID for importing into an existing album.')]
  #[CLI\Option(name: 'create-album', description: 'Create a new album and set the album title.')]
  #[CLI\Option(name: 'uid', description: 'The user ID if creating a new album. Defaults to 1.')]
  #[CLI\Option(name: 'promote', description: 'Promote to the homepage if this is a new album.')]
  #[CLI\Usage(name: 'photos:import /my-photos --nid=1', description: 'Import image files in /my-photos to an existing photo album with node ID 1.')]
  #[CLI\Usage(name: 'photos:import /my-photos --create-album="Album title" --promote=1', description: 'Import image files in /my-photos to a new album and promote to the homepage.')]
  public function import(
    ?string $directory = '',
    ?array $options = [
      'nid' => NULL,
      'create-album' => NULL,
      'uid' => 1,
      'promote' => NULL,
    ],
  ): void {
    if (!empty($directory) && is_dir($directory) && (!empty($options['nid']) || !empty($options['create-album']))) {
      $album_id = $options['nid'] ? intval($options['nid']) : NULL;

      if (!empty($options['create-album'])) {
        $uid = $options['uid'];
        // Create a new photo album.
        /** @var \Drupal\node\NodeInterface $node */
        $fields = [
          'type' => 'photos',
          'title' => $options['create-album'],
          'uid' => $uid,
        ];
        if ($options['promote']) {
          $fields['promote'] = 1;
        }
        $album = $this->entityTypeManager->getStorage('node')->create($fields);
        $album->save();
        $album_id = $album->id();
        $this->logger->notice(dt('Album: %album created.', [
          '%album' => $album->label(),
        ]));
      }
      $count = 0;
      if (!empty($album_id)) {
        $this->logger->notice('Importing image files...');
        // @note this could be a batch.
        $count = $this->photosUpload->directoryImport('drush', $album_id, $directory);
      }

      $this->logger()->notice(dt('%count images imported.', [
        '%count' => $count,
      ]));
    }
    else {
      if (empty($options['nid'])) {
        $this->logger()->alert(dt('No album id.'));
      }
      if (empty($directory) || !file_exists($directory)) {
        $this->logger()->alert(dt('Could not find directory: %s', [
          '%s' => $directory,
        ]));
      }
    }
  }

}
