<?php

namespace Drupal\photos;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Html;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Utility\Error;
use Drupal\views\Views;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Help manage photos.
 */
class PhotosManager implements PhotosManagerInterface {

  use MessengerTrait;
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The Current User object.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * The logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The current request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\Core\Image\ImageFactory $image_factory
   *   The image factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The current request stack.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $connection,
    AccountInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    FileUrlGeneratorInterface $file_url_generator,
    ImageFactory $image_factory,
    LoggerInterface $logger,
    RequestStack $request_stack,
    StreamWrapperManagerInterface $stream_wrapper_manager,
    TimeInterface $time,
  ) {
    $this->configFactory = $config_factory;
    $this->connection = $connection;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUrlGenerator = $file_url_generator;
    $this->imageFactory = $image_factory;
    $this->logger = $logger;
    $this->requestStack = $request_stack;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->time = $time;
  }

  /**
   * {@inheritdoc}
   */
  public function getCount(string $type, int $id = 0): int {
    $db = $this->connection;
    $count = 0;
    switch ($type) {
      case 'user_album':
      case 'user_image':
      case 'site_album':
      case 'site_image':
        $count = $db->query("SELECT value FROM {photos_count} WHERE cid = :cid AND type = :type", [
          ':cid' => $id,
          ':type' => $type,
        ])->fetchField();
        break;

      case 'node_album':
        $count = $db->query("SELECT count FROM {photos_album} WHERE album_id = :album_id", [
          ':album_id' => $id,
        ])->fetchField();
    }
    return $count;
  }

  /**
   * {@inheritdoc}
   */
  public function setCount(string $type, int $id = 0) {
    $db = $this->connection;
    $requestTime = $this->time->getRequestTime();
    switch ($type) {
      case 'user_image':
        $count = $db->query('SELECT count(p.id) FROM {photos_image_field_data} p WHERE p.uid = :id', [
          ':id' => $id,
        ])->fetchField();
        $db->merge('photos_count')
          ->keys([
            'cid' => $id,
            'type' => $type,
          ])
          ->fields([
            'value' => $count,
            'changed' => $requestTime,
          ])
          ->execute();
        // Clear cache tags.
        Cache::invalidateTags(['photos:image:user:' . $id]);
        break;

      case 'user_album':
        $count = $db->query('SELECT count(p.album_id) FROM {photos_album} p INNER JOIN {node_field_data} n ON p.album_id = n.nid WHERE n.uid = :uid', [
          ':uid' => $id,
        ])->fetchField();

        $db->merge('photos_count')
          ->keys([
            'cid' => $id,
            'type' => $type,
          ])
          ->fields([
            'value' => $count,
            'changed' => $requestTime,
          ])
          ->execute();
        // Clear cache tags.
        Cache::invalidateTags(['photos:album:user:' . $id]);
        break;

      case 'site_album':
        $count = $db->query('SELECT COUNT(album_id) FROM {photos_album}')->fetchField();
        $db->merge('photos_count')
          ->keys([
            'cid' => 0,
            'type' => $type,
          ])
          ->fields([
            'value' => $count,
            'changed' => $requestTime,
          ])
          ->execute();
        break;

      case 'site_image':
        $count = $db->query('SELECT COUNT(id) FROM {photos_image_field_data}')->fetchField();
        $db->merge('photos_count')
          ->keys([
            'cid' => 0,
            'type' => $type,
          ])
          ->fields([
            'value' => $count,
            'changed' => $requestTime,
          ])
          ->execute();
        // Clear cache tags.
        Cache::invalidateTags(['photos:image:recent']);
        break;

      case 'node_album':
        $count = $db->query("SELECT COUNT(id) FROM {photos_image_field_data} WHERE album_id = :album_id", [
          ':album_id' => $id,
        ])->fetchField();
        $db->update('photos_album')
          ->fields([
            'count' => $count,
          ])
          ->condition('album_id', $id)
          ->execute();
        break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function resetCount(bool $cron = FALSE) {
    $this->setCount('site_album');
    $this->setCount('site_image');
    // @todo add variable for this?
    $time = $cron ? 7200 : 0;
    $config = $this->configFactory->getEditable('photos.settings');
    $cron_last_run = $config->get('cron_last_run') ? $config->get('cron_last_run') : 0;
    $request_time = $this->time->getRequestTime();
    // @todo add variable for limit?
    $limit = 50;
    if (($request_time - $cron_last_run) > $time) {
      $db = $this->connection;
      $processed_uids = [];
      // Add any users that do not have a count yet.
      $result = $db->query("SELECT UNIQUE uid FROM {users} u
           LEFT JOIN {photos_count} pc ON (pc.type = 'user_image' OR pc.type = 'user_album') AND pc.cid = u.uid
           WHERE u.uid != 0 AND (pc.cid IS NULL OR pc.changed = 0) LIMIT " . $limit);
      foreach ($result as $t) {
        if (isset($processed_uids[$t->uid])) {
          continue;
        }
        $processed_uids[$t->uid] = $t->uid;
        $this->setCount('user_image', $t->uid);
        $this->setCount('user_album', $t->uid);
      }
      // Refresh existing user counts starting with the oldest.
      $sql = "SELECT UNIQUE uid FROM {users} u
           LEFT JOIN {photos_count} pc ON pc.cid = u.uid
           WHERE u.uid != 0";
      if (!empty($processed_uids)) {
        $sql .= " AND uid NOT IN (" . implode(', ', $processed_uids) . ")";
      }
      $sql .= " ORDER BY pc.changed ASC LIMIT " . $limit;
      $result = $db->query($sql);
      foreach ($result as $t) {
        if (isset($processed_uids[$t->uid])) {
          continue;
        }
        $this->setCount('user_image', $t->uid);
        $this->setCount('user_album', $t->uid);
      }
      // Refresh album counts.
      $result = $db->query("SELECT album_id FROM {photos_album} pa
          LEFT JOIN {photos_count} pc ON pc.cid = pa.album_id AND pc.type = 'node_album'
          ORDER BY pc.changed ASC LIMIT " . $limit);
      foreach ($result as $t) {
        $this->setCount('node_album', $t->album_id);
      }
      $config->set('cron_last_run', $request_time)->save();
    }
    // @todo create a batch and button to rebuild counts manually if needed.
  }

  /**
   * {@inheritdoc}
   */
  public function getAlbumCover(int $album_id, int $cover_id = 0, bool $uri_only = FALSE): array {
    $cover = [];
    if (!$cover_id) {
      // Check album for cover fid.
      $db = $this->connection;
      $cover_id = $db->query("SELECT cover_id FROM {photos_album} WHERE album_id = :album_id", [
        ':album_id' => $album_id,
      ])->fetchField();
    }
    // If id is still empty.
    if (empty($cover_id)) {
      // Cover not set, select an image from the album.
      $db = $this->connection;
      $query = $db->select('photos_image_field_data', 'p');
      $query->fields('p', ['id']);
      $query->condition('p.album_id', $album_id);
      $cover_id = $query->execute()->fetchField();
    }
    if ($cover_id) {
      // Load image.
      $photos_image = NULL;
      try {
        /** @var \Drupal\photos\PhotosImageInterface $photos_image */
        $photos_image = $this->entityTypeManager
          ->getStorage('photos_image')
          ->load($cover_id);
      }
      catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
        Error::logException($this->logger, $e);
      }
      if ($photos_image) {
        if ($uri_only) {
          if ($photos_image->hasField('field_image') && $photos_image->field_image->entity) {
            $cover = $photos_image->field_image->entity->getFileUri();
          }
        }
        else {
          $cover = $this->entityTypeManager
            ->getViewBuilder('photos_image')
            ->view($photos_image, 'cover');
        }
      }
    }
    return $cover;
  }

  /**
   * {@inheritdoc}
   */
  public function setAlbumCover(int $album_id, int $cover_id = 0) {
    // Update cover.
    $db = $this->connection;
    $db->update('photos_album')
      ->fields([
        'cover_id' => $cover_id,
      ])
      ->condition('album_id', $album_id)
      ->execute();
    // Clear node and views cache.
    Cache::invalidateTags([
      'node:' . $album_id,
      'photos:album:' . $album_id,
      'photos_image_list',
    ]);
    $this->messenger()->addMessage($this->t('Cover successfully set.'));
  }

  /**
   * {@inheritdoc}
   */
  public function getAlbumImages(int $album_id, int $limit = 10): array {
    $images = [];
    // Prepare query.
    $get_field = $this->requestStack->getCurrentRequest()->query->get('field');
    $column = $get_field ? Html::escape($get_field) : '';
    $get_sort = $this->requestStack->getCurrentRequest()->query->get('sort');
    $sort = $get_sort ? Html::escape($get_sort) : '';
    $term = $this->orderValue($column, $sort, $limit, [
      'column' => 'p.weight',
      'sort' => 'asc',
    ]);
    $photos_image_storage = $this->entityTypeManager->getStorage('photos_image');
    // Query images in this album.
    $results = $photos_image_storage->getQuery()
      ->accessCheck()
      ->condition('album_id', $album_id)
      ->sort('weight', 'asc')
      ->range(0, $term['limit'])
      ->execute();
    // Prepare images.
    foreach ($results as $id) {
      $photosImage = $photos_image_storage->load($id);
      $images[] = [
        'photos_image' => $photosImage,
      ];
    }
    return $images;
  }

  /**
   * {@inheritdoc}
   */
  public function userAlbumCount(): array {
    $user = $this->currentUser;
    $user_roles = $user->getRoles();
    $t['create'] = $this->getCount('user_album', $user->id());
    // @todo upgrade path? Check D7 role id and convert pnum variables as needed.
    $role_limit = 0;
    $t['total'] = 20;
    // Check highest role limit.
    $config = $this->configFactory->get('photos.settings');
    foreach ($user_roles as $role) {
      if ($config->get('photos_pnum_' . $role)
        && $config->get('photos_pnum_' . $role) > $role_limit) {
        $role_limit = $config->get('photos_pnum_' . $role);
      }
    }
    if ($role_limit > 0) {
      $t['total'] = $role_limit;
    }

    $t['remain'] = ($t['total'] - $t['create']);
    if ($user->id() != 1 && $t['remain'] <= 0) {
      $t['rest'] = 1;
    }
    return $t;
  }

  /**
   * {@inheritdoc}
   */
  public function getPager(int $entity_id, int $id, string $type = 'album_id'): array {
    $db = $this->connection;
    $query = $db->select('photos_image_field_data', 'p');
    $query->innerJoin('node_field_data', 'n', 'n.nid = p.album_id');
    $query->fields('p', ['id', 'album_id']);
    $query->fields('n', ['title']);

    // Default order by id.
    $order_conditions = [];
    $order_conditions[] = ['column' => 'p.id', 'sort' => 'DESC'];
    if ($type == 'album_id') {
      // Viewing album.
      $photos_album_list_view = Views::getView('photos_album');
      if ($photos_album_list_view) {
        // Get sort order from views.
        $photos_album_list_view->initDisplay();
        $display = $photos_album_list_view->getDisplay();
        $display_sorts = $display->getOption('sorts');
        if (!empty($display_sorts)) {
          $order_conditions = [];
          foreach ($display_sorts as $sort_criterion) {
            $order_conditions[] = $this->orderValueChange($sort_criterion['field'], $sort_criterion['order']);
          }
        }
      }
      $query->condition('p.album_id', $id);
    }
    elseif ($type == 'uid') {
      // Viewing all user images.
      $query->condition('p.uid', $id);
    }
    $add_default = TRUE;
    foreach ($order_conditions as $order) {
      $query->orderBy($order['column'], $order['sort']);
      if ($order['column'] == 'p.id') {
        $add_default = FALSE;
      }
    }
    // Additional sort order / default.
    if ($add_default) {
      $query->orderBy('p.id', 'DESC');
    }
    $results = $query->execute();

    $stop = $pager['prev'] = $pager['next'] = 0;
    $previousImageId = NULL;
    $photosImageStorage = $this->entityTypeManager->getStorage('photos_image');
    $photosImageViewBuilder = $this->entityTypeManager->getViewBuilder('photos_image');
    // @todo needs more testing with different translations.
    foreach ($results as $result) {
      if ($stop == 1) {
        if ($result->id != $entity_id) {
          /** @var \Drupal\photos\PhotosImageInterface $photosImage */
          $photosImage = $photosImageStorage->load($result->id);
          $image_view = $photosImageViewBuilder->view($photosImage, 'pager');
          $pager['nextView'] = $image_view;
          // Next image.
          $pager['nextUrl'] = Url::fromRoute('entity.photos_image.canonical', [
            'node' => $result->album_id,
            'photos_image' => $photosImage->id(),
          ])->toString();
          break;
        }
      }
      if ($result->id == $entity_id) {
        $photosImage = $photosImageStorage->load($result->id);
        $image_view = $photosImageViewBuilder->view($photosImage, 'pager');
        $pager['currentView'] = $image_view;
        $stop = 1;
      }
      else {
        if (!$stop) {
          $previousImageId = $result->id;
        }
      }
      $pager['albumTitle'] = $result->title;
    }
    if ($previousImageId) {
      $photosImage = $photosImageStorage->load($previousImageId);
      $image_view = $photosImageViewBuilder->view($photosImage, 'pager');
      $pager['prevView'] = $image_view;
      // Previous image.
      $pager['prevUrl'] = Url::fromRoute('entity.photos_image.canonical', [
        'node' => $id,
        'photos_image' => $photosImage->id(),
      ])->toString();
    }

    // @todo theme photos_pager with options for image and no-image.
    return $pager;
  }

  /**
   * {@inheritdoc}
   */
  public function orderValue(string $field, string $sort, int $limit, array $default = []): array {
    // @todo update default to check album default!
    $t = [];
    $default_order = ['column' => 'p.id', 'sort' => 'desc'];
    if (!$field && !$sort) {
      $t['order'] = !$default ? $default_order : $default;
    }
    else {
      if (!$t['order'] = $this->orderValueChange($field, $sort)) {
        $t['order'] = !$default ? $default_order : $default;
      }
    }
    if ($limit) {
      $get_limit = $this->requestStack->getCurrentRequest()->query->get('limit');
      if ($get_limit && !$show = intval($get_limit)) {
        $get_destination = $this->requestStack->getCurrentRequest()->query->get('destination');
        if ($get_destination) {
          $str = $get_destination;
          if (preg_match('/.*limit=(\d*).*/i', $str, $mat)) {
            $show = intval($mat[1]);
          }
        }
      }
      $t['limit'] = $show ?? $limit;
    }

    return $t;
  }

  /**
   * {@inheritdoc}
   */
  public function orderValueChange(string $field, string $sort): array {
    // @note timestamp is deprecated, but may exist
    // if albums are migrated from a previous version.
    // @todo add support for other views sort criteria as needed.
    $array = [
      'weight' => 'p.weight',
      'title' => 'p.title',
      'timestamp' => 'p.id',
      'id' => 'p.id',
      'changed' => 'p.changed',
      'created' => 'p.created',
      'comments' => 'c.comment_count',
      'visits' => 'v.value',
      'filesize' => 'f.filesize',
    ];
    $array1 = [
      'asc' => 'asc',
      'ASC' => 'asc',
      'desc' => 'desc',
      'DESC' => 'desc',
    ];
    if (isset($array[$field]) && isset($array1[$sort])) {
      return [
        'column' => $array[$field],
        'sort' => $array1[$sort],
      ];
    }
    else {
      // Default if values not found.
      return [
        'column' => 'p.id',
        'sort' => 'desc',
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function view(int $fid, string $style_name = '', array $variables = []): array {
    // @todo this should be updated to use display modes now.
    $image = $this->load($fid);
    if (!$image) {
      return [];
    }
    if (isset($variables['title'])) {
      $image->title = $variables['title'];
    }
    if (!$style_name) {
      // Get thumbnail image style from admin settings.
      $image_sizes = $this->configFactory->get('photos.settings')->get('photos_size');
      $style_name = key($image_sizes);
    }
    if (!$style_name) {
      // Fallback on default thumbnail style.
      $style_name = 'thumbnail';
    }
    // Check scheme and prep image.
    $scheme = $this->streamWrapperManager->getScheme($image->uri);
    $uri = $image->uri;
    // If private create temporary derivative.
    if ($scheme == 'private') {
      // @todo this is currently broken. Try using display modes or
      //   re-implement hook_file_download.
      $url = $this->derivative($fid, $uri, $style_name, $scheme);
    }
    else {
      // Public and all other images.
      $style = $this->entityTypeManager->getStorage('image_style')->load($style_name);
      $url = $style->buildUrl($uri);
    }
    // Build image render array.
    $title = $image->title ?? '';
    $alt = $image->alt ?? $title;
    $image_render_array = [
      '#theme' => 'image',
      '#uri' => $url,
      '#alt' => $alt,
      '#title' => $title,
    ];
    if (isset($variables['href'])) {
      $image_render_array = [
        '#type' => 'link',
        '#title' => $image_render_array,
        '#url' => Url::fromUri('base:' . $variables['href']),
      ];
    }

    return $image_render_array;
  }

  /**
   * {@inheritdoc}
   */
  public function derivative(int $fid, string $uri, string $style_name, string $scheme = 'private'): string {
    // Load the image style configuration entity.
    /** @var \Drupal\image\ImageStyleInterface $style */
    $style = $this->entityTypeManager->getStorage('image_style')->load($style_name);

    // Create URI with fid_{fid}.
    $pathInfo = pathinfo($uri);
    $ext = strtolower($pathInfo['extension']);
    // Set temporary file destination.
    $destination = $scheme . '://photos/tmp_images/' . $style_name . '/image_' . $fid . '.' . $ext;
    // Create image file.
    $style->createDerivative($uri, $destination);

    // Generate an itok.
    $itok = $style->getPathToken($uri);

    // Return URL.
    return $this->fileUrlGenerator->generateAbsoluteString($destination) . '?itok=' . $itok;
  }

  /**
   * Load image file and album data.
   */
  public function load($fid) {
    // Query image data.
    // @todo check access. Is ->addTag('node_access') needed here? If so,
    //   rewrite query. I think access is already checked before we get here.
    $db = $this->connection;
    // @note currently legacy mode requires default field_image.
    $image = $db->query('SELECT f.fid, f.uri, f.filemime, f.created, f.filename, n.title as node_title, a.data, u.uid, u.name, p.*
      FROM {file_managed} f
      INNER JOIN {photos_image__field_image} i ON i.field_image_target_id = f.fid
      INNER JOIN {photos_image_field_data} p ON p.revision_id = i.revision_id
      INNER JOIN {node_field_data} n ON p.album_id = n.nid
      INNER JOIN {photos_album} a ON a.album_id = n.nid
      INNER JOIN {users_field_data} u ON f.uid = u.uid
      WHERE i.field_image_target_id = :fid', [':fid' => $fid])->fetchObject();
    // Set image height and width.
    if (!isset($image->height) && isset($image->uri)) {
      // The image.factory service will check if our image is valid.
      $image_info = $this->imageFactory->get($image->uri);
      if ($image_info->isValid()) {
        $image->width = $image_info->getWidth();
        $image->height = $image_info->getHeight();
      }
      else {
        $image->width = $image->height = NULL;
      }
    }
    return $image;
  }

}
