<?php

namespace Drupal\photos;

/**
 * Defines an interface for the photo manager.
 */
interface PhotosManagerInterface {

  /**
   * Return number of albums or photos.
   *
   * @param string $type
   *   The type of statistics.
   * @param int $id
   *   The album id or image entity id.
   */
  public function getCount(string $type, int $id = 0): int;

  /**
   * Set count for type and id.
   *
   * @param string $type
   *   The type of statistics.
   * @param int $id
   *   The album id or image entity id.
   */
  public function setCount(string $type, int $id = 0);

  /**
   * Reset count.
   *
   * @param bool $cron
   *   If this is being called from cron.
   */
  public function resetCount(bool $cron = FALSE);

  /**
   * Get album cover.
   *
   * @param int $album_id
   *   The album id.
   * @param int $cover_id
   *   The photos_image entity id.
   * @param bool $uri_only
   *   Return image URI only if available (used for photos album media).
   *
   * @return array
   *   Render array.
   */
  public function getAlbumCover(int $album_id, int $cover_id = 0, bool $uri_only = FALSE): array;

  /**
   * Set album cover.
   *
   * @param int $album_id
   *   The album id.
   * @param int $cover_id
   *   The photos_image entity id.
   */
  public function setAlbumCover(int $album_id, int $cover_id = 0);

  /**
   * Get album images.
   *
   * @param int $album_id
   *   The album id.
   * @param int $limit
   *   The number of images to return.
   *
   * @todo this looks like it is only in the re-arrange page, maybe move there?
   */
  public function getAlbumImages(int $album_id, int $limit = 10): array;

  /**
   * Tracks number of albums created and number of albums allowed.
   *
   * @todo add $user param?
   */
  public function userAlbumCount(): array;

  /**
   * Gets the page for this image.
   *
   * @param int $entity_id
   *   The PhotosImage id.
   * @param int $id
   *   The pager id: album_id or uid.
   * @param string $type
   *   The type of pager: album_id or uid.
   *
   * @return array
   *   The photos image pager data or render array.
   */
  public function getPager(int $entity_id, int $id, string $type = 'album_id'): array;

  /**
   * Query helper: sort order and limit.
   *
   * @param string $field
   *   The field to sort by.
   * @param string $sort
   *   The sort order.
   * @param int $limit
   *   The limit.
   * @param array $default
   *   The default sort order.
   *
   * @todo remove this?
   */
  public function orderValue(string $field, string $sort, int $limit, array $default = []): array;

  /**
   * Extends photos order value.
   *
   * @param string $field
   *   The field to sort by.
   * @param string $sort
   *   The sort order.
   *
   * @todo remove this?
   */
  public function orderValueChange(string $field, string $sort): array;

  /**
   * Return render array to view image.
   *
   * @param int $fid
   *   The {file_managed}.fid.
   * @param string $style_name
   *   The image style machine name.
   * @param array $variables
   *   (Optional) variables to override image defaults:
   *   - 'title': image title and alt if alt is empty.
   *   - 'href': image link href.
   *
   * @return array
   *   Render array for image view.
   */
  public function view(int $fid, string $style_name = '', array $variables = []): array;

  /**
   * Generate image style derivatives and return image file URL.
   *
   * Originally added to create private image style derivatives.
   *
   * @param int $fid
   *   The {file_managed}.fid.
   * @param string $uri
   *   File URI.
   * @param string $style_name
   *   Image style name.
   * @param string $scheme
   *   File system scheme.
   *
   * @return string
   *   The image URL.
   */
  public function derivative(int $fid, string $uri, string $style_name, string $scheme = 'private'): string;

}
