<?php

namespace Drupal\photos;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystem;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Image\ImageFactory;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StreamWrapper\PrivateStream;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Error;
use Drupal\Core\Utility\Token;
use Drupal\file\FileInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\file\FileUsage\FileUsageInterface;
use Drupal\file\Validation\FileValidatorInterface;
use Drupal\media\Entity\Media;
use Drupal\photos\Entity\PhotosImage;
use Psr\Log\LoggerInterface;

/**
 * Functions to help with uploading images to albums.
 */
class PhotosUpload implements PhotosUploadInterface {

  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The Current User object.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystem
   */
  protected $fileSystem;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * The file validator.
   *
   * @var \Drupal\file\Validation\FileValidatorInterface
   */
  protected FileValidatorInterface $fileValidator;

  /**
   * The image factory.
   *
   * @var \Drupal\Core\Image\ImageFactory
   */
  protected $imageFactory;

  /**
   * The logger instance.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The photos manager service.
   *
   * @var \Drupal\photos\PhotosManagerInterface
   */
  protected $photosManager;

  /**
   * The stream wrapper manager.
   *
   * @var \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface
   */
  protected $streamWrapperManager;

  /**
   * The token replacement instance.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * Transliteration service.
   *
   * @var \Drupal\Component\Transliteration\TransliterationInterface
   */
  protected $transliteration;

  /**
   * The entity storage for the 'file' entity type.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $fileStorage;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_manager
   *   The entity manager service.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   * @param \Drupal\Core\File\FileSystem $file_system
   *   The file system service.
   * @param \Drupal\file\FileUsage\FileUsageInterface $file_usage
   *   File usage service.
   * @param \Drupal\file\Validation\FileValidatorInterface $file_validator
   *   The file validator.
   * @param \Drupal\Core\Image\ImageFactory $image_factory
   *   The image factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\photos\PhotosManagerInterface $photos_manager
   *   The photos manager service.
   * @param \Drupal\Core\StreamWrapper\StreamWrapperManagerInterface $stream_wrapper_manager
   *   The stream wrapper manager.
   * @param \Drupal\Core\Utility\Token $token
   *   The token replacement instance.
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   The transliteration service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $connection,
    AccountInterface $current_user,
    EntityFieldManagerInterface $entity_field_manager,
    EntityTypeManagerInterface $entity_manager,
    FileRepositoryInterface $file_repository,
    FileSystem $file_system,
    FileUsageInterface $file_usage,
    FileValidatorInterface $file_validator,
    ImageFactory $image_factory,
    LoggerInterface $logger,
    MessengerInterface $messenger,
    ModuleHandlerInterface $module_handler,
    PhotosManagerInterface $photos_manager,
    StreamWrapperManagerInterface $stream_wrapper_manager,
    Token $token,
    TransliterationInterface $transliteration,
  ) {
    $this->configFactory = $config_factory;
    $this->connection = $connection;
    $this->currentUser = $current_user;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_manager;
    $this->fileRepository = $file_repository;
    $this->fileSystem = $file_system;
    $this->fileUsage = $file_usage;
    $this->fileValidator = $file_validator;
    $this->imageFactory = $image_factory;
    $this->logger = $logger;
    $this->messenger = $messenger;
    $this->moduleHandler = $module_handler;
    $this->photosManager = $photos_manager;
    $this->streamWrapperManager = $stream_wrapper_manager;
    $this->token = $token;
    $this->transliteration = $transliteration;

    $this->fileStorage = $this->entityTypeManager->getStorage('file');
  }

  /**
   * {@inheritdoc}
   */
  public function cleanTitle($title = '') {
    $config = $this->configFactory->get('photos.settings');
    if ($config->get('photos_clean_title')) {
      // Remove extension.
      $title = pathinfo($title, PATHINFO_FILENAME);
      // Replace dash and underscore with spaces.
      $title = preg_replace("/[\-_]/", " ", $title);
      // Trim leading and trailing spaces.
      $title = trim($title);
    }
    return $title;
  }

  /**
   * {@inheritdoc}
   */
  public function path($schemeType = 'default') {
    $fileConfig = $this->configFactory->get('system.file');
    $path[] = 'photos';
    switch ($schemeType) {
      case 'private':
        $scheme = 'private';
        break;

      case 'public':
        $scheme = 'public';
        break;

      case 'default':
      default:
        $scheme = $fileConfig->get('default_scheme');
        break;
    }
    $dirs = [];
    // Prepare directory.
    foreach ($path as $folder) {
      $dirs[] = $folder;
      $finalPath = $scheme . '://' . implode('/', $dirs);
      if (!$this->fileSystem->prepareDirectory($finalPath, FileSystemInterface::CREATE_DIRECTORY)) {
        return FALSE;
      }
    }
    if ($finalPath) {
      // Make sure the path does not end with a forward slash.
      $finalPath = rtrim($finalPath, '/');
    }
    return $finalPath;
  }

  /**
   * {@inheritdoc}
   */
  public function getScheme(int $nid): string {
    $scheme = 'default';
    if ($this->moduleHandler->moduleExists('photos_access')) {
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
      if (isset($node->photos_privacy['viewid'])) {
        $album_viewid = $node->photos_privacy['viewid'];
        if ($album_viewid > 0) {
          // Check for private file path.
          // @todo add support for other schemes?
          if (PrivateStream::basePath()) {
            $scheme = 'private';
          }
          else {
            // Set warning message.
            $this->messenger->addWarning($this->t('Warning: image
                files can still be accessed by visiting the direct URL. For
                better security, ask your website admin to setup a private
                file path.'));
          }
        }
      }
    }
    return $scheme;
  }

  /**
   * {@inheritdoc}
   */
  public function saveImage(FileInterface $file) {
    $config = $this->configFactory->get('photos.settings');
    // @todo maybe pass file object and array of other vars.
    if ($file->id() && isset($file->album_id)) {
      $fid = $file->id();
      $albumId = $file->album_id;
      $album_photo_limit = $config->get('album_photo_limit');

      $photo_count = $this->connection->query("SELECT count FROM {photos_album} WHERE album_id = :album_id", [
        ':album_id' => $albumId,
      ])->fetchField();

      if ($album_photo_limit && ($photo_count >= $album_photo_limit)) {
        return FALSE;
      }

      // Prep image title.
      if (!empty($file->title)) {
        $title = $file->title;
      }
      else {
        // Cleanup filename and use as title.
        $title = $this->cleanTitle($file->getFilename());
      }

      // Create photos_image entity.
      /** @var \Drupal\Core\Image\Image $image */
      $image = $this->imageFactory->get($file->getFileUri());
      $defaultWeight = $this->connection->select('photos_image_field_data', 'i')
        ->fields('i', ['weight'])
        ->condition('i.album_id', $albumId)
        ->orderBy('i.weight', 'DESC')
        ->execute()->fetchField();
      if ($image->isValid()) {
        $newPhotosImageEntity = [
          'album_id' => $albumId,
          'title' => $title,
          'weight' => $file->weight ?? ($defaultWeight + 1),
          'description' => $file->des ?? '',
        ];
        if ($this->currentUser->isAnonymous()) {
          $newPhotosImageEntity['uid'] = $file->getOwnerId();
        }
        // Check if photos_image has default field_image.
        $uploadField = $config->get('multi_upload_default_field');
        $uploadFieldParts = explode(':', $uploadField);
        $field = $uploadFieldParts[0] ?? 'field_image';
        $allBundleFields = $this->entityFieldManager->getFieldDefinitions('photos_image', 'photos_image');
        if (isset($allBundleFields[$field])) {
          $fieldType = $allBundleFields[$field]->getType();
          if ($fieldType == 'image') {
            $newPhotosImageEntity[$field] = [
              'target_id' => $fid,
              'alt' => $title,
              'title' => $title,
              'width' => $image->getWidth(),
              'height' => $image->getHeight(),
            ];
          }
          else {
            // Check media fields.
            if ($fieldType == 'entity_reference') {
              $mediaField = $uploadFieldParts[1] ?? '';
              $mediaBundle = $uploadFieldParts[2] ?? '';
              if ($mediaField && $mediaBundle) {
                // Create new media entity.
                $values = [
                  'bundle' => $mediaBundle,
                  'uid' => $this->currentUser->id(),
                ];
                $values[$mediaField] = [
                  'target_id' => $file->id(),
                ];
                $media = Media::create($values);
                // @todo media name?
                $media->setName('Photo ' . $file->id())->setPublished()->save();
                // Set photos_image media reference field.
                $newPhotosImageEntity[$field] = [
                  'target_id' => $media->id(),
                ];
              }
            }
          }
        }
        $photosImage = PhotosImage::create($newPhotosImageEntity);
        try {
          $photosImage->save();
          if ($photosImage && $photosImage->id()) {
            if (isset($fieldType) && $fieldType == 'image') {
              // Move image to correct directory.
              $fieldThirdPartySettings = $allBundleFields[$field]->getThirdPartySettings('filefield_paths');
              if (!empty($fieldThirdPartySettings) && $fieldThirdPartySettings['enabled']) {
                // Get path from filefield_paths.
                $tokenData = [
                  'file' => $file,
                  $photosImage->getEntityTypeId() => $photosImage,
                ];
                $name = $file->getFilename();
                if (!empty($fieldThirdPartySettings['file_name']['value'])) {
                  $name = filefield_paths_process_string($fieldThirdPartySettings['file_name']['value'], $tokenData, $fieldThirdPartySettings['file_name']['options']);
                }
                // Process filepath.
                $path = filefield_paths_process_string($fieldThirdPartySettings['file_path']['value'], $tokenData, $fieldThirdPartySettings['file_path']['options']);
                $fileUri = $this->streamWrapperManager
                  ->normalizeUri($this->streamWrapperManager->getScheme($file->getFileUri()) . '://' . $path . DIRECTORY_SEPARATOR . $name);
              }
              else {
                // Get path from field settings.
                $fieldSettings = $allBundleFields[$field]->getSettings();
                $uploadLocation = $fieldSettings['file_directory'];
                $uploadLocation = PlainTextOutput::renderFromHtml($this->token->replace($uploadLocation, []));
                $uploadLocation = $this->streamWrapperManager->getScheme($file->getFileUri()) . '://' . $uploadLocation;
                $this->fileSystem->prepareDirectory($uploadLocation, FileSystemInterface::CREATE_DIRECTORY);
                $fileUri = "{$uploadLocation}/{$file->getFilename()}";
                $fileUri = $this->fileSystem->getDestinationFilename($fileUri, FileExists::Rename);
                // Move the file.
                $this->fileSystem->move($file->getFileUri(), $fileUri, FileExists::Error);
              }
              // Set the correct URI and save the file.
              $file->setFileUri($fileUri);
              $file->save();
            }
            if ($config->get('photos_user_count_cron')) {
              $user = $this->currentUser;
              $this->photosManager->setCount('user_image', ($photosImage->getOwnerId() ? $photosImage->getOwnerId() : $user->id()));
              $this->photosManager->setCount('node_album', $albumId);
            }
            // Save file and add file usage.
            $this->fileUsage->add($file, 'photos', 'node', $albumId);
            // Check admin setting for maximum image resolution.
            if ($photos_size_max = $config->get('photos_size_max')) {
              // Will scale image if needed.
              $validators = [
                'FileImageDimensions' => [
                  'maxDimensions' => $photos_size_max,
                ],
              ];
              /** @var \Symfony\Component\Validator\ConstraintViolationListInterface $violations */
              $this->fileValidator->validate($file, $validators);
              // Get new height and width for field values.
              if (isset($fieldType)) {
                $image = $this->imageFactory->get($file->getFileUri());
                if ($fieldType == 'image') {
                  $image_files = $photosImage->get($field)->getValue();
                  $image_files[0]['height'] = $image->getHeight();
                  $image_files[0]['width'] = $image->getWidth();
                  // Save new height and width.
                  $photosImage->set($field, $image_files);
                  $photosImage->save();
                }
                else {
                  if (isset($media) && isset($mediaField)) {
                    $image_files = $media->get($mediaField)->getValue();
                    $image_files[0]['height'] = $image->getHeight();
                    $image_files[0]['width'] = $image->getWidth();
                    // Save new height and width.
                    $media->set($mediaField, $image_files);
                    $media->save();
                  }
                }
              }
            }
            return TRUE;
          }
        }
        catch (EntityStorageException $e) {
          Error::logException($this->logger, $e);
        }
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function saveExistingMedia($mediaId, $albumId) {
    $config = $this->configFactory->get('photos.settings');
    /** @var \Drupal\media\MediaInterface $mediaItem */
    $mediaItem = NULL;
    try {
      $mediaItem = $this->entityTypeManager->getStorage('media')
        ->load($mediaId);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
    }
    if ($mediaItem) {
      $defaultWeight = $this->connection->select('photos_image_field_data', 'i')
        ->fields('i', ['weight'])
        ->condition('i.album_id', $albumId)
        ->orderBy('i.weight', 'DESC')
        ->execute()->fetchField();
      $newPhotosImageEntity = [
        'album_id' => $albumId,
        'title' => $mediaItem->getName(),
        'weight' => ($defaultWeight + 1),
      ];
      // Check default media field.
      $uploadField = $config->get('multi_upload_default_field');
      $uploadFieldParts = explode(':', $uploadField);
      $field = $uploadFieldParts[0] ?? 'field_image';
      $allBundleFields = $this->entityFieldManager->getFieldDefinitions('photos_image', 'photos_image');
      if (isset($allBundleFields[$field])) {
        $fieldType = $allBundleFields[$field]->getType();
        if ($fieldType == 'entity_reference') {
          $mediaField = $uploadFieldParts[1] ?? '';
          $mediaBundle = $uploadFieldParts[2] ?? '';
          if ($mediaField && $mediaBundle) {
            // Set photos_image media reference field.
            $newPhotosImageEntity[$field] = [
              'target_id' => $mediaId,
            ];
          }
          // Save PhotosImage entity.
          $photosImage = $this->entityTypeManager->getStorage('photos_image')->create($newPhotosImageEntity);
          try {
            $photosImage->save();
            if ($photosImage && $photosImage->id()) {
              if ($config->get('photos_user_count_cron')) {
                $user = $this->currentUser;
                $this->photosManager->setCount('user_image', ($photosImage->getOwnerId() ? $photosImage->getOwnerId() : $user->id()));
                $this->photosManager->setCount('node_album', $albumId);
              }
              return TRUE;
            }
          }
          catch (EntityStorageException $e) {
            Error::logException($this->logger, $e);
          }
        }
      }
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function unzip($source, array $params, $scheme = 'default') {
    $fileConfig = $this->configFactory->get('system.file');
    $file_count = 0;
    $photo_count = 0;
    if (isset($params['photo_count'])) {
      $photo_count = $params['photo_count'];
    }
    $album_photo_limit = NULL;
    if (isset($params['album_photo_limit'])) {
      $album_photo_limit = $params['album_photo_limit'];
    }
    if (version_compare(PHP_VERSION, '5') >= 0) {
      if (!is_file($source)) {
        $this->messenger->addMessage($this->t('Compressed file does not exist, please check the path: @src', [
          '@src' => $source,
        ]));
        return 0;
      }
      $fileType = $this->getFileExtensions('array');
      $zip = new \ZipArchive();
      // Get relative path.
      $default_scheme = $fileConfig->get('default_scheme');
      $relative_path = $this->fileSystem->realpath($default_scheme . "://") . '/';
      $source = str_replace($default_scheme . '://', $relative_path, $source);
      // Open zip archive.
      if ($zip->open($source) === TRUE) {
        for ($x = 0; $x < $zip->numFiles; ++$x) {
          $image = $zip->statIndex($x);
          $image_name = basename($image['name']);
          $filename_parts = explode('.', $image['name']);
          $ext = end($filename_parts);
          $ext = strtolower($ext);
          if (in_array($ext, $fileType)) {
            if ($album_photo_limit && ($photo_count >= $album_photo_limit)) {
              $this->messenger->addWarning($this->t('Maximum number of photos reached for this album.'));
              break;
            }
            $path = $this->fileSystem->createFilename($image_name, $this->path($scheme));
            if ($temp_file = $this->fileRepository->writeData($zip->getFromIndex($x), $path)) {
              // Update file values.
              $temp_file->album_id = $params['album_id'];
              $temp_file->nid = $params['nid'];
              // Use image file name as title.
              $temp_file->title = $image_name;
              $temp_file->des = $params['description'] ?? '';
              // Prepare file entity.
              $file = $temp_file;
              try {
                // Save image.
                $file->save();
                if ($this->saveImage($file)) {
                  $file_count++;
                }
              }
              catch (EntityStorageException $e) {
                Error::logException($this->logger, $e);
              }
            }
          }
        }
        $zip->close();
        // Delete zip file.
        $this->fileSystem->delete($source);
      }
      else {
        $this->messenger->addWarning($this->t('Compressed file does not exist, please try again: @src', [
          '@src' => $source,
        ]));
      }
    }

    return $file_count;
  }

  /**
   * {@inheritdoc}
   */
  public function getFileExtensions(string $type = 'string', string $separator = ', '): string|array {
    $extensions = '';
    $allowed_extensions = $this->configFactory->get('photos.settings')->get('allowed_file_extensions') ?? [];
    if (!empty($allowed_extensions)) {
      switch ($type) {
        case 'string':
          if ($separator != ', ') {
            $extensions = str_replace(', ', $separator, $allowed_extensions);
          }
          break;

        case 'array':
          $extensions = explode(', ', $allowed_extensions);
          break;
      }
    }
    return $extensions;
  }

  /**
   * {@inheritdoc}
   */
  public function directoryImport(string $trigger, int $album_id = 0, string $directory = ''): int {
    $count = 0;
    $config = $this->configFactory->get('photos.settings');
    $backup_directory = '';
    $limit = 0;
    $op = 'copy';
    if ($trigger == 'cron') {
      $cron_import = $config->get('cron_import');
      $directory = $cron_import['directory'] ?? '';
      $album_id = $cron_import['album'];
      $op = $cron_import['directory_op'];
      $backup_directory = $cron_import['backup_directory'] ?? '';
      $limit = $cron_import['limit'] ?? 0;
    }
    if (!empty($directory) && file_exists($directory) && !empty($album_id)) {
      // Get album data.
      /** @var \Drupal\node\NodeInterface $album */
      $album = $this->entityTypeManager
        ->getStorage('node')
        ->load($album_id);
      if ($album) {
        $uid = $album->getOwnerId();
        // Other settings.
        $scheme = $this->getScheme($album_id);
        // Get the files.
        $files = $this->directoryGetFiles($directory);
        if ($limit) {
          $process_files = array_slice($files, 0, $limit);
        }
        else {
          $process_files = $files;
        }
        foreach ($process_files as $dir_file) {
          $processed = $this->processDirectoryFile($album_id, $uid, $scheme, $dir_file, $op, $backup_directory);
          if ($processed) {
            $count = $count + $processed;
          }
        }
        if ($count > 0) {
          // Update user image count.
          $this->photosManager->setCount('user_image', $uid);
          // Update album image count.
          $this->photosManager->setCount('node_album', $album_id);
        }
      }
    }
    return $count;
  }

  /**
   * {@inheritdoc}
   */
  public function directoryGetFiles(string $directory): array {
    $config = $this->configFactory->get('photos.settings');
    $allow_zip = $config->get('photos_upzip') ? '|zip|ZIP' : '';
    $allowed_file_extensions = $this->getFileExtensions('string', '|');
    if (!empty($allowed_file_extensions)) {
      // Include lowercase and uppercase extensions.
      $allowed_file_extensions = strtolower($allowed_file_extensions) . '|' . strtoupper($allowed_file_extensions);
    }
    $file_extensions = $allowed_file_extensions . $allow_zip;

    // Get the files.
    return $this->fileSystem->scanDirectory($directory, '/^.*\.(' . $file_extensions . ')$/');
  }

  /**
   * {@inheritdoc}
   */
  public function processDirectoryFile(int $album_id, int $uid, string $scheme, object $dir_file, string $op, string $backup_directory = ''): int {
    $processed = 0;
    $file_processed = '';
    // Prepare directory.
    $photos_path = $this->path($scheme);
    $photos_name = $dir_file->filename;
    $file_uri = $this->fileSystem
      ->getDestinationFilename($photos_path . '/' . $photos_name, FileExists::Rename);
    switch ($op) {
      case 'copy':
        // Leave original file in the source directory.
        $file_processed = $this->fileSystem->copy($dir_file->uri, $file_uri);
        break;

      case 'delete':
        // Move from the import directory to the Drupal file system.
        $file_processed = $this->fileSystem->move($dir_file->uri, $file_uri);
        break;

      case 'move':
        if (!empty($backup_directory)) {
          $backup_file_uri = $this->fileSystem
            ->getDestinationFilename($backup_directory . '/' . $photos_name, FileExists::Rename);
          // Copy into the Drupal file system.
          $file_processed = $this->fileSystem->copy($dir_file->uri, $file_uri);
          // Move to the backup directory.
          $this->fileSystem->move($dir_file->uri, $backup_file_uri);
        }
        break;
    }

    $ext = mb_substr($dir_file->uri, -3);
    if ($ext != 'zip' && $ext != 'ZIP') {
      // Regular image file.
      if ($file_processed) {
        // Save file to album. Include title and description.
        /** @var \Drupal\Core\Image\Image $image */
        $image = $this->imageFactory->get($file_uri);
        if ($image->isValid() && $image->getWidth()) {
          // Create a file entity.
          /** @var \Drupal\file\FileInterface $file */
          $file = $this->fileStorage->create([
            'uri' => $file_uri,
            'uid' => $uid,
            'status' => FileInterface::STATUS_PERMANENT,
            'album_id' => $album_id,
            'nid' => $album_id,
            'filename' => $photos_name,
            'filesize' => $image->getFileSize(),
            'filemime' => $image->getMimeType(),
          ]);

          try {
            $file->save();
            if ($this->saveImage($file)) {
              // File successfully saved to the album.
              $processed = 1;
            }
          }
          catch (EntityStorageException $e) {
            Error::logException($this->logger, $e);
          }
        }
      }
    }
    else {
      // Handle zip files.
      if ($file_processed) {
        $params = [];
        $params['album_id'] = $album_id;
        $params['nid'] = $album_id;
        $params['description'] = '';
        $params['title'] = $dir_file->filename;
        // @todo large zip files could fail here.
        if ($file_count = $this->unzip($file_processed, $params, $scheme)) {
          // Count image files in unzipped directory.
          $processed = $processed + $file_count;
          // @todo delete zip directory after successful import?
        }
      }
    }
    return $processed;
  }

}
