<?php

namespace Drupal\photos;

use Drupal\file\FileInterface;

/**
 * Photos upload and file management functions.
 */
interface PhotosUploadInterface {

  /**
   * Rename file with random name.
   *
   * @param string $title
   *   The file name.
   *
   * @return string
   *   The new name.
   */
  public function cleanTitle($title = '');

  /**
   * Temporary file path.
   *
   * The image file path is now handled in the field settings. This is used
   * if needed before the field settings are triggered.
   *
   * @param string $schemeType
   *   URI scheme.
   *
   * @return string
   *   A path where image files will be saved.
   */
  public function path($schemeType = 'default');

  /**
   * Check URI scheme and warn if private files can be accessed.
   *
   * @param int $nid
   *   The album node ID.
   *
   * @return string
   *   The scheme.
   */
  public function getScheme(int $nid): string;

  /**
   * Attach image file to PhotosImage entity.
   *
   * @param \Drupal\file\FileInterface $file
   *   The file entity.
   *
   * @return bool
   *   TRUE if image saved successfully.
   */
  public function saveImage(FileInterface $file);

  /**
   * Attach media to PhotosImage entity.
   *
   * @param int $mediaId
   *   The media entity_id.
   * @param int $albumId
   *   The album entity_id.
   *
   * @return bool
   *   TRUE if media saved successfully.
   */
  public function saveExistingMedia($mediaId, $albumId);

  /**
   * Unzip archive of image files.
   *
   * @param string $source
   *   The zip file location.
   * @param array $params
   *   Array of additional parameters like album id.
   * @param string $scheme
   *   The file scheme.
   *
   * @return int
   *   Uploaded files count.
   */
  public function unzip($source, array $params, $scheme = 'default');

  /**
   * Get and format file extensions.
   *
   * @param string $type
   *   The type of list to return: array or string.
   * @param string $separator
   *   The string separator.
   *
   * @return string|array
   *   The formatted list of allowed file extensions.
   */
  public function getFileExtensions(string $type = 'string', string $separator = ', '): string|array;

  /**
   * Import image files from a directory.
   *
   * @param string $trigger
   *   Trigger: cron, drush, form or test.
   * @param int $album_id
   *   The album node ID.
   * @param string $directory
   *   The directory path.
   *
   * @return int
   *   Number of images imported.
   */
  public function directoryImport(string $trigger, int $album_id = 0, string $directory = ''): int;

  /**
   * Scan directory for image files.
   *
   * @param string $directory
   *   The directory path.
   *
   * @return array
   *   An array of files.
   */
  public function directoryGetFiles(string $directory): array;

  /**
   * Process a single file for a directory import.
   *
   * @param int $album_id
   *   The album node ID.
   * @param int $uid
   *   The user ID.
   * @param string $scheme
   *   The URI scheme.
   * @param object $dir_file
   *   The file from the source directory.
   * @param string $op
   *   What to do with the original file:
   *   - copy: leave the file in the directory.
   *   - move: move the file to a different (backup) directory.
   *   - delete: move the file out of the directory.
   * @param string $backup_directory
   *   The backup directory (cron directory import feature).
   *
   * @return int
   *   The number of files processed.
   */
  public function processDirectoryFile(int $album_id, int $uid, string $scheme, object $dir_file, string $op, string $backup_directory = ''): int;

}
