<?php

namespace Drupal\Tests\photos\Functional;

use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\language\Entity\ContentLanguageSettings;
use Drupal\Tests\BrowserTestBase;

/**
 * Test creating a new album, adding an image and updating the image.
 *
 * @group photos
 */
class AddPhotosAlbumImageTest extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'content_translation',
    'node',
    'field',
    'field_ui',
    'file',
    'image',
    'comment',
    'photos',
    'photos_views_test',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The user account for testing.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * The admin user account for testing.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminAccount;

  /**
   * The location of the fixtures files.
   *
   * @var string
   */
  protected $path;

  /**
   * The file system.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The photos upload handler.
   *
   * @var \Drupal\photos\PhotosUploadInterface
   */
  protected $photosUpload;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a new language.
    ConfigurableLanguage::createFromLangcode('es')->save();
    ContentLanguageSettings::create([
      'target_entity_type_id' => 'photos_image',
      'target_bundle' => 'photos_image',
    ])
      ->setThirdPartySetting('content_translation', 'enabled', TRUE)
      ->save();

    // Create user with permissions to edit own photos.
    $this->account = $this->drupalCreateUser([
      'create photo',
      'create photos content',
      'delete own photo',
      'edit own photo',
      'edit own photos content',
      'view photo',
    ]);
    $this->drupalLogin($this->account);

    // Create admin account that can update field settings.
    $this->adminAccount = $this->drupalCreateUser([
      'administer photos_image fields',
      'edit any photo',
      'translate any entity',
      'view photo',
    ]);

    // @note add more test images if we need more controlled tests.
    $this->path = 'core/tests/fixtures/files';

    $this->fileSystem = $this->container->get('file_system');
    $this->photosUpload = $this->container->get('photos.upload');
  }

  /**
   * Test adding an image to an album and accessing the image edit page.
   */
  public function testAccessPhotosImageEditForm(): void {

    // Create a test album node.
    $albumTitle = $this->randomMachineName();
    /** @var \Drupal\Core\Entity\ContentEntityStorageInterface $storage */
    $nodeStorage = $this->container->get('entity_type.manager')->getStorage('node');
    $album = $nodeStorage->create([
      'type' => 'photos',
      'title' => $albumTitle,
    ]);
    $album->save();

    // Get test image file.
    $testPhotoUri = \Drupal::service('extension.list.module')->getPath('photos') . '/tests/images/photos-test-picture.jpg';

    // Post image upload form.
    $edit = [
      'files[images_0]' => $this->fileSystem->realpath($testPhotoUri),
      'title_0' => 'Test photo title',
      'des_0' => 'Test photos description',
    ];
    $this->drupalGet('node/' . $album->id() . '/photos');
    $this->submitForm($edit, 'Confirm upload');

    // Get album images.
    $albumImages = \Drupal::service('photos.manager')->getAlbumImages($album->id(), 1);
    /** @var \Drupal\photos\PhotosImageInterface $photosImage */
    $photosImage = $albumImages[0]['photos_image'];
    $this->assertEquals($edit['title_0'], $photosImage->getTitle());

    // Access image edit page.
    $this->drupalGet('photos/' . $photosImage->getAlbumId() . '/' . $photosImage->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Post image edit form.
    $edit = [
      'title[0][value]' => 'Test new title',
    ];
    $this->submitForm($edit, 'Save');

    // Confirm that image title has been updated.
    /** @var \Drupal\Core\Entity\ContentEntityStorageInterface $storage */
    $storage = $this->container->get('entity_type.manager')->getStorage('photos_image');
    // Must explicitly clear cache to see new title.
    // @see https://www.drupal.org/project/drupal/issues/3040878
    $storage->resetCache([$photosImage->id()]);
    /** @var \Drupal\photos\PhotosImageInterface $photosImage */
    $photosImage = $storage->load($photosImage->id());
    $this->assertEquals($edit['title[0][value]'], $photosImage->getTitle());

    // Test recent albums content overview.
    $this->drupalGet('photos');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains($albumTitle);

    // Check image file directory settings.
    /** @var \Drupal\file\FileInterface $image_file */
    $image_file = $photosImage->field_image->entity;
    $this->assertStringContainsString('photos/images', $image_file->getFileUri());
    // Change image field file directory settings.
    $this->drupalLogin($this->adminAccount);
    $edit = [
      'settings[file_directory]' => 'custom_directory/images',
    ];
    $this->drupalGet('admin/structure/photos/fields/photos_image.photos_image.field_image');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm($edit, 'Save settings');

    // Upload new image.
    $this->drupalLogin($this->account);
    $edit = [
      'files[images_0]' => $this->fileSystem->realpath($testPhotoUri),
      'title_0' => 'Test2 photo title',
      'des_0' => 'Test2 photos description',
    ];
    $this->drupalGet('node/' . $album->id() . '/photos');
    $this->submitForm($edit, 'Confirm upload');

    // Test directory import features.
    // @see Drupal\KernelTests\Core\File\ScanDirectoryTest
    $count = $this->photosUpload->directoryImport('test', $album->id(), $this->path);
    $this->assertGreaterThan(5, $count);

    // Get album images.
    $albumImages = \Drupal::service('photos.manager')->getAlbumImages($album->id(), 2);
    /** @var \Drupal\photos\PhotosImageInterface $photosImage */
    $photosImage = $albumImages[1]['photos_image'];
    $this->assertStringContainsString('custom_directory/images', $photosImage->field_image->entity->getFileUri());

    // Test photo translate form.
    $enTitle = $photosImage->getTitle();
    $this->drupalLogin($this->adminAccount);
    $this->drupalGet('photos/' . $photosImage->getAlbumId() . '/' . $photosImage->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
    $edit = [
      'title[0][value]' => 'Test translated title',
      'description[0][value]' => 'Test translated description',
    ];
    $this->drupalGet('photos/' . $photosImage->getAlbumId() . '/' . $photosImage->id() . '/translations/add/en/es');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm($edit, 'Save');

    // Refresh entities.
    $albumId = $photosImage->getAlbumId();
    $photosId = $photosImage->id();
    $album = $nodeStorage->load($albumId);
    $photosImage = $storage->load($photosId);
    $esPhotosImage = $photosImage->getTranslation('es');
    $this->assertEquals($edit['title[0][value]'], $esPhotosImage->getTitle());
    $this->assertEquals($enTitle, $photosImage->getTitle());

    // View node page.
    $this->drupalGet('node/' . $albumId);
    $this->assertSession()->statusCodeEquals(200);

    // View photo page.
    $this->drupalGet('photos/' . $albumId . '/' . $photosId);
    $this->assertSession()->statusCodeEquals(200);

    // Delete album.
    $album->delete();

    // View node page.
    $this->drupalGet('node/' . $albumId);
    $this->assertSession()->statusCodeEquals(404);

    // View photo page.
    $this->drupalGet('photos/' . $albumId . '/' . $photosId);
    $this->assertSession()->statusCodeEquals(404);

    // Make sure this user cannot import photos.
    $count = $this->photosUpload->directoryImport('test', $albumId, $this->path);
    $this->assertEquals(0, $count);

  }

}
