<?php

declare(strict_types=1);

namespace Drupal\Tests\php_password\Kernel;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DrupalKernel;
use Drupal\Core\Password\PasswordInterface;
use Drupal\Core\Site\Settings;
use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\TestWith;

/**
 * Tests PhpPasswordServiceProvider.
 *
 * @group php_password
 * @coversDefaultClass \Drupal\php_password\PhpPasswordServiceProvider
 */
final class PhpPasswordServiceProviderTest extends KernelTestBase {

  protected static $modules = [
    'php_password',
  ];

  public function register(ContainerBuilder $container) {
    // This sets the password algorithm and cost for performance reasons.
    parent::register($container);
    // Let's undo that so we can test the module's functionality.
    $container->getDefinition('password')->setArguments([]);
  }

  #[TestWith([\PASSWORD_DEFAULT, 'password'])]
  #[TestWith([\PASSWORD_DEFAULT, PasswordInterface::class])]
  #[TestWith([\PASSWORD_ARGON2I, 'password'])]
  #[TestWith([\PASSWORD_ARGON2I, PasswordInterface::class])]
  #[TestWith([\PASSWORD_ARGON2ID, 'password'])]
  #[TestWith([\PASSWORD_ARGON2ID, PasswordInterface::class])]
  public function testAlgorithm(string $algorithm, string $identifier): void {
    // Should default to PASSWORD_DEFAULT.
    $password = $this->container->get($identifier);
    \assert($password instanceof PasswordInterface);
    $hash = $password->hash('i can haz access?');
    $info = \password_get_info($hash);
    self::assertEquals(\PASSWORD_DEFAULT, $info['algo']);

    // Now write a services file with password.algorithm set.
    $services_file = $this->siteDirectory . '/services.yml';
    $services = [];
    if (\is_file($services_file)) {
      $yaml = \file_get_contents($services_file) ?: '';
      $services = Yaml::decode($yaml);
    }
    $services['parameters']['password.algorithm'] = $algorithm;
    $services['parameters']['password.options'] = [];
    \file_put_contents($services_file, Yaml::encode($services));
    $service_files = Settings::get('container_yamls', []);
    $this->setSetting('container_yamls', \array_unique([
      ...$service_files,
      $services_file
    ]));
    $this->container = $this->container->get('kernel')->rebuildContainer();

    $password = $this->container->get($identifier);
    \assert($password instanceof PasswordInterface);
    $hash = $password->hash('i can haz access?');
    $info = \password_get_info($hash);
    self::assertEquals($algorithm, $info['algo']);
  }

}
