<?php

declare(strict_types=1);

namespace Drupal\pie_chart\Controller;

use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Node Pie Chart routes.
 */
final class NodePieChartController implements ContainerInjectionInterface {

  /**
   * The database connection service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new NodePieChartController.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Builds the response for node data.
   */
  public function getNodeData() {
    $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $content_types = [];
    $node_counts = [];

    foreach ($node_types as $type) {
      $type_id = $type->id();
      $type_label = $type->label();
      $content_types[] = $type_label;

      // Query to get the node count for the current content type.
      $query = $this->database->select('node_field_data', 'nfd')
        ->condition('nfd.type', $type_id)
        ->condition('nfd.status', 1)
        ->countQuery();
      $count = $query->execute()->fetchField();

      $node_counts[$type_label] = $count;
    }

    $build = [
      '#theme' => 'node-data',
      '#content_types' => $content_types,
      '#node_counts' => $node_counts,
      '#attached' => [
        'drupalSettings' => [
          'pie_chart' => [
            'node_counts' => $node_counts,
          ],
        ],
      ],
    ];

    return $build;
  }

  /**
   * Builds the response for user data.
   */
  public function getUserData() {
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $counts = [];

    foreach ($roles as $role_id => $role) {
      $query = $this->entityTypeManager->getStorage('user')->getQuery()
        ->accessCheck(FALSE)
        ->condition('status', 1)
        ->condition('roles', $role_id);
      $count = $query->count()->execute();

      $counts[$role->label()] = $count;
    }

    $build = [
      '#theme' => 'user-data',
      '#counts' => $counts,
      '#attached' => [
        'library' => ['pie_chart/drupal.pie_chart'],
        'drupalSettings' => [
          'userCounts' => $counts,
        ],
      ],
    ];
    return $build;
  }

}
