<?php

namespace Drupal\pie_chart\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides a Custom Pie Chart Block with Name/Value pairs.
 *
 * @Block(
 *   id = "custom_pie_chart_block",
 *   admin_label = @Translation("Custom Pie Chart Block"),
 * )
 */
class CustomPieChartBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'items_raw' => "Apples|10\nBananas|20\nCherries|15",
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form['items_raw'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Name/Value pairs'),
      '#description' => $this->t('Enter one pair per line as Name|Value. Example: Apples|10'),
      '#default_value' => $this->configuration['items_raw'] ?? '',
      '#rows' => 7,
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['items_raw'] = (string) $form_state->getValue('items_raw');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $pairs_text = (string) ($this->configuration['items_raw'] ?? '');
    $data = [];

    foreach (preg_split('/\r?\n/', $pairs_text) as $line) {
      $line = trim($line);
      if ($line === '') {
        continue;
      }
      if (strpos($line, '|') === FALSE) {
        continue;
      }
      [$name, $value] = array_map('trim', explode('|', $line, 2));
      if ($name === '') {
        continue;
      }
      // Cast numeric-like values to integer; ignore invalid numbers.
      if ($value === '' || !is_numeric($value)) {
        continue;
      }
      $data[$name] = (int) $value;
    }

    return [
      '#theme' => 'node-block',
      '#cache' => [
        'max-age' => 0,
      ],
      '#attached' => [
        'library' => ['pie_chart/drupal.pie_chart'],
        'drupalSettings' => [
          'customblock' => $data,
        ],
      ],
    ];
  }

}
