<?php

namespace Drupal\pie_chart\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

/**
 * Provides a Pie Chart Block.
 *
 * @Block(
 *   id = "node_pie_chart_block",
 *   admin_label = @Translation("Node Pie Chart Block"),
 * )
 */
class NodePieChartBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a NodePieChartBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The database service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected Connection $database,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Load all content types.
    $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $node_counts = [];

    // Fetch node count for each content type.
    foreach ($node_types as $type) {
      $type_id = $type->id();
      $type_label = $type->label();

      // Query to count published nodes of the current content type.
      $count = $this->database
        ->select('node_field_data', 'nfd')
        ->condition('nfd.type', $type_id)
        ->condition('nfd.status', 1)
        ->countQuery()
        ->execute()
        ->fetchField();

      $node_counts[$type_label] = $count;
    }

    return [
      '#theme' => 'node-block',
      '#cache' => [
        'max-age' => 0,
      ],
      '#attached' => [
        'library' => ['pie_chart/drupal.pie_chart'],
        'drupalSettings' => [
          'fromblock' => $node_counts,
        ],
      ],
    ];
  }

}
