<?php

namespace Drupal\pillarshield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\node\NodeInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;

/**
 * Client wrapper for communicating with the Pillarshield SaaS API.
 *
 * This service is intentionally thin: it only:
 * - Resolves the API key via the Key module.
 * - Flattens content into a single "content" string (if asked).
 * - Adds a "meta" block with non-PII context for audit logging.
 * - Sends the HTTP request and normalizes the response.
 *
 * All governance logic (block/allow/override) happens in the hook layer.
 */
class PillarshieldClient {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The current user proxy.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuid;

  /**
   * Constructs a new PillarshieldClient.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user proxy.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelInterface $logger,
    KeyRepositoryInterface $key_repository,
    AccountProxyInterface $current_user,
    UuidInterface $uuid
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->keyRepository = $key_repository;
    $this->currentUser = $current_user;
    $this->uuid = $uuid;
  }

  /**
   * Legacy helper: builds content from entity fields then calls checkContent().
   *
   * This is still here for possible reuse, but our form validator now calls
   * checkContent() directly with content built from the submitted form values.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node being saved.
   * @param string[] $field_names
   *   Fields to include.
   *
   * @return array
   *   Normalized response array.
   */
  public function checkNode(NodeInterface $node, array $field_names) {
    $chunks = [];
    foreach ($field_names as $field_name) {
      if ($node->hasField($field_name)) {
        $value = $node->get($field_name)->value ?? NULL;
        if (is_string($value) && $value !== '') {
          $chunks[] = $value;
        }
      }
    }

    $content = implode("\n\n", $chunks);

    // Fallback to title if content is somehow empty.
    if ($content === '' && method_exists($node, 'getTitle')) {
      $content = (string) $node->getTitle();
    }

    return $this->checkContent($node, $content);
  }

  /**
   * Checks the given raw content string with the Pillarshield API.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node being saved (for meta only).
   * @param string $content
   *   The pre-flattened content string to audit.
   *
   * @return array
   *   An array with keys:
   *   - status: int|null
   *   - body: array|null
   *   - error: string|null
   */
  public function checkContent(NodeInterface $node, string $content) {
    $config = $this->configFactory->get('pillarshield.settings');
    $endpoint = $config->get('api_endpoint');
    $key_id = $config->get('api_key_id');

    if (!$endpoint || !$key_id) {
      $this->logger->warning('Pillarshield API endpoint or Key ID not configured.');
      return [
        'status' => NULL,
        'body' => NULL,
        'error' => 'Configuration missing',
      ];
    }

    // Resolve the API key using the Key module.
    $key = $this->keyRepository->getKey($key_id);
    if (!$key) {
      $this->logger->error('Pillarshield Key "@id" not found.', ['@id' => $key_id]);
      return [
        'status' => NULL,
        'body' => NULL,
        'error' => 'Key not found',
      ];
    }

    $api_key = $key->getKeyValue();
    if (!$api_key) {
      $this->logger->error('Pillarshield Key "@id" has no value.', ['@id' => $key_id]);
      return [
        'status' => NULL,
        'body' => NULL,
        'error' => 'Key has no value',
      ];
    }

    // If there's really nothing to scan, log and soft-fail.
    if ($content === '') {
      $this->logger->warning('Pillarshield: No content collected for node @id of type @type.', [
        '@id' => $node->id(),
        '@type' => $node->bundle(),
      ]);

      return [
        'status' => NULL,
        'body' => NULL,
        'error' => 'No content to audit',
      ];
    }

    // Build meta: CMS-agnostic context for audit logging (no raw PII).
    $environment = getenv('PILLARSHIELD_ENV') ?: 'prod';
    $editor_id = (int) $this->currentUser->id();
    $langcode = $node->language()->getId();
    $revision_id = method_exists($node, 'getRevisionId') ? (int) $node->getRevisionId() : null;
    $uuid = method_exists($node, 'uuid') ? (string) $node->uuid() : null;
    $bundle = method_exists($node, 'bundle') ? (string) $node->bundle() : null;
    $entity_type = $node->getEntityTypeId();
    // SAFE context for SaaS audit logs (no emails, no names, no usernames).
    $context = [
      // Existing SaaS-supported page fields:
      'page_type' => $entity_type . ':' . $bundle,
      'page_title' => (string) ($node->label() ?? ''),
      'content_context' => 'ui:save',

      // WHO (actor):
      'actor_type' => 'drupal_user',
      'actor_id' => (string) $editor_id,
      'actor_roles' => array_values($this->currentUser->getRoles()),
      'actor_label' => 'uid:' . $editor_id, // safe pseudonym

      // WHERE / WHAT:
      'content_type' => $entity_type,
      'content_id' => (string) $node->id(),
      'content_langcode' => (string) $langcode,
      'content_revision_id' => $revision_id ? (string) $revision_id : null,
      'content_uuid' => $uuid,
      'content_bundle' => $bundle,
    ];
    try {
      // Canonical URL to the content (absolute, no query string).
      $context['content_url'] = $node->toUrl('canonical', ['absolute' => TRUE])->toString();
    } catch (\Throwable $e) {
      // Fail-open: URL resolution errors shouldn't block saves.
    }
    $meta = [
      'cms' => 'drupal',
      'environment' => $environment,
      'entity_type' => $entity_type,
      'request_source' => 'ui:save',
      'bundle' => $bundle,
    ];

    $payload = [
      'content' => $content,
      'api_key' => $api_key,
      'context' => $context,
      'meta' => $meta,
    ];

    try {
      $response = $this->httpClient->request('POST', $endpoint, [
        'headers' => [
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
        'timeout' => 10,
        'http_errors' => FALSE,
      ]);

      $status = $response->getStatusCode();
      $raw = (string) $response->getBody();
      $body = json_decode($raw, TRUE);

      if ($body === NULL && $raw !== '' && json_last_error() !== JSON_ERROR_NONE) {
        $this->logger->error('Pillarshield API returned non-JSON response: @raw', ['@raw' => $raw]);
        return [
          'status' => $status,
          'body' => NULL,
          'error' => 'Invalid JSON from API',
        ];
      }

      return [
        'status' => $status,
        'body' => $body,
        'error' => NULL,
      ];
    }
    catch (\Exception $e) {
      $this->logger->error('Pillarshield API request failed: @message', [
        '@message' => $e->getMessage(),
      ]);

      return [
        'status' => NULL,
        'body' => NULL,
        'error' => $e->getMessage(),
      ];
    }
  }

}
