<?php

namespace Drupal\Tests\pillarshield\Kernel;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;
use Drupal\user\Entity\Role;
use Drupal\user\Entity\User;

/**
 * @group pillarshield
 */
class PillarshieldGovernanceTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'node',
    'pillarshield',
  ];

  /**
   * Pillarshield client stub.
   *
   * @var object
   */
  protected $pillarshieldClientStub;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install schemas.
    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('system', ['sequences']);

    // Install node config (content types, etc).
    $this->installConfig(['node']);

    // Create a basic "article" content type.
    $type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $type->save();

    // Configure Pillarshield defaults.
    $editable = $this->container->get('config.factory')->getEditable('pillarshield.settings');
    $editable
      ->set('api_endpoint', 'https://example.com/pillarshield/check')
      ->set('api_key', 'test-key')
      ->set('enable_governance', TRUE)
      ->set('allow_save_without_api', TRUE)
      ->set('enabled_content_types', ['article'])
      // Scan only the title to keep things simple in tests.
      ->set('fields_per_type', [
        'article' => ['title'],
      ])
      ->save();

    // Replace the real Pillarshield client with a stub we can control.
    $this->pillarshieldClientStub = new class() {
      public $result = [
        'status' => 200,
        'body' => [],
        'error' => NULL,
      ];

      public function checkNode(NodeInterface $node, array $field_names) {
        return $this->result;
      }
    };

    $this->container->set('pillarshield.client', $this->pillarshieldClientStub);
  }

  /**
   * Helper to get messenger messages of a specific type.
   *
   * @param string $type
   *   The message type: 'status', 'warning', or 'error'.
   *
   * @return string[]
   *   Array of plain-text messages.
   */
  protected function getMessagesOfType(string $type): array {
    $messenger = $this->container->get('messenger');
    $messages = $messenger->messagesByType($type);
    $strings = [];

    foreach ($messages as $message) {
      // messagesByType() returns arrays with 'message' as a MarkupInterface
      // or string; cast to string for comparison.
      $strings[] = (string) $message;
    }

    return $strings;
  }

  /**
   * Test that compliant content (200) saves normally.
   */
  public function testCompliantContentSavesNormally(): void {
    // Simulate a compliant API response.
    $this->pillarshieldClientStub->result = [
      'status' => 200,
      'body' => [
        'message' => 'OK',
      ],
      'error' => NULL,
    ];

    // Simple user without special permissions.
    $user = User::create([
      'name' => 'editor',
      'status' => 1,
    ]);
    $user->save();
    $this->setCurrentUser($user);

    $node = Node::create([
      'type' => 'article',
      'title' => 'Compliant content',
    ]);

    $node->save();

    $this->assertNotEmpty($node->id(), 'Node was saved successfully.');

    $errors = $this->getMessagesOfType('error');
    $this->assertEmpty($errors, 'No governance errors were reported.');
  }

  /**
   * Test that a violation (406) blocks save when there is no override.
   */
  public function testViolationBlocksSaveWithoutOverride(): void {
    $this->pillarshieldClientStub->result = [
      'status' => 406,
      'body' => [
        'message' => 'Content violates governance rules.',
        'violations' => [
          [
            'field' => 'title',
            'code' => 'PROHIBITED_TERM',
            'detail' => "Content contains prohibited term: 'Guaranteed'",
          ],
        ],
      ],
      'error' => NULL,
    ];

    // User without override permission.
    $user = User::create([
      'name' => 'editor_no_override',
      'status' => 1,
    ]);
    $user->save();
    $this->setCurrentUser($user);

    $node = Node::create([
      'type' => 'article',
      'title' => 'Guaranteed best price',
    ]);

    $exceptionThrown = FALSE;
    try {
      $node->save();
    }
    catch (EntityStorageException $e) {
      $exceptionThrown = TRUE;
    }

    $this->assertTrue($exceptionThrown, 'Save was blocked by Pillarshield on violation.');

    $errors = $this->getMessagesOfType('error');
    $this->assertNotEmpty($errors, 'Governance errors were reported.');
    $this->assertStringContainsString('violates governance rules', implode(' | ', $errors));
    $this->assertStringContainsString('PROHIBITED_TERM', implode(' | ', $errors));
  }

  /**
   * Test that a violation (406) can be overridden when allowed.
   */
  public function testViolationAllowsSaveWithOverridePermissionAndFlag(): void {
    $this->pillarshieldClientStub->result = [
      'status' => 406,
      'body' => [
        'message' => 'Content violates governance rules.',
        'violations' => [
          [
            'type' => 'PROHIBITED_TERM',
            'message' => "Content contains prohibited term.",
            'effective_severity' => 'error',
          ],
        ],
      ],
      'error' => NULL,
    ];

    // Create a role with the override permission.
    $role = Role::create([
      'id' => 'pillarshield_override',
      'label' => 'Pillarshield Override',
    ]);
    $role->grantPermission('bypass pillarshield governance');
    $role->save();

    // User with the override role.
    $user = User::create([
      'name' => 'overrider',
      'status' => 1,
    ]);
    $user->addRole($role->id());
    $user->save();
    $this->setCurrentUser($user);

    $node = Node::create([
      'type' => 'article',
      'title' => 'Guaranteed best price',
    ]);

    // Simulate the user having checked the "Save anyway" checkbox on the form
    // by setting the transient property directly.
    $node->pillarshieldOverride = TRUE;

    // This time, the save should succeed despite the 406.
    $node->save();
    $this->assertNotEmpty($node->id(), 'Node was saved despite violations due to override.');

    $warnings = $this->getMessagesOfType('warning');
    $errors = $this->getMessagesOfType('error');

    $this->assertNotEmpty($warnings, 'Governance violations were reported as warnings.');
    $this->assertEmpty($errors, 'No errors were raised when override was used.');
    $this->assertStringContainsString('violates governance rules', implode(' | ', $warnings));
    $this->assertStringContainsString('You chose to save this content despite Pillarshield governance violations', implode(' | ', $warnings));
  }

}
