<?php

namespace Drupal\pillarshield\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Pillarshield governance.
 */
class PillarshieldSettingsForm extends ConfigFormBase {

  /**
   * The bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $bundleInfo;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    /** @var \Drupal\pillarshield\Form\PillarshieldSettingsForm $instance */
    $instance = parent::create($container);
    $instance->bundleInfo = $container->get('entity_type.bundle.info');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pillarshield.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pillarshield_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pillarshield.settings');
    $form['#tree'] = TRUE;
    // ----------------------------------------------------------------------
    // API configuration.
    // ----------------------------------------------------------------------
    $form['api'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Pillarshield API'),
    ];

    $form['api']['api_endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('API endpoint'),
      '#default_value' => $config->get('api_endpoint') ?: '',
      '#required' => TRUE,
      '#description' => $this->t('Base endpoint for the Pillarshield governance API (POST).'),
    ];

    // Build a list of available Key entities for the API key.
    $key_options = [];
    $key_storage = $this->entityTypeManager->getStorage('key');
    $keys = $key_storage->loadMultiple();
    foreach ($keys as $key) {
      /** @var \Drupal\key\Entity\KeyInterface $key */
      $key_options[$key->id()] = $key->label() . ' (' . $key->id() . ')';
    }

    $form['api']['api_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('API key'),
      '#options' => $key_options,
      '#default_value' => $config->get('api_key_id') ?: NULL,
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select a Key to use -'),
      '#description' => $this->t('Select the Key entity that stores your Pillarshield API key. Manage keys at Configuration → System → Keys.'),
    ];

    $form['api']['test_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'pillarshield-test-result'],
      'markup' => [
        '#type' => 'markup',
        '#markup' => '<p style="margin:0;">' . $this->t('Click “Test governance connection” to verify endpoint + key + response format.') . '</p>',
      ],
    ];

    $form['api']['test_connection'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test governance connection'),
      '#submit' => ['::testConnectionSubmit'],
      '#limit_validation_errors' => [],
    ];
    // ----------------------------------------------------------------------
    // Governance behavior.
    // ----------------------------------------------------------------------
    $form['governance'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Governance behavior'),
    ];

    $form['governance']['enable_governance'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Pillarshield governance'),
      '#default_value' => $config->get('enable_governance') ?? TRUE,
      '#description' => $this->t('If disabled, Pillarshield will not inspect content before it is saved.'),
    ];

    $form['governance']['allow_save_without_api'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow saves when the Pillarshield API is unreachable'),
      '#default_value' => $config->get('allow_save_without_api') ?? TRUE,
      '#description' => $this->t('If disabled, content saves will fail when the API cannot be reached.'),
    ];

    // ----------------------------------------------------------------------
    // Content types & fields configuration.
    // ----------------------------------------------------------------------
    $form['content_types'] = [
      '#type' => 'details',
      '#title' => $this->t('Content types and fields'),
      '#open' => TRUE,
    ];

    // Node bundles.
    $bundles = $this->bundleInfo->getBundleInfo('node');
    $options = [];
    foreach ($bundles as $id => $definition) {
      $options[$id] = $definition['label'] ?? $id;
    }

    $form['content_types']['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Governed content types'),
      '#options' => $options,
      '#default_value' => $config->get('enabled_content_types') ?: [],
      '#description' => $this->t('Pillarshield checks will run for these content types.'),
    ];

    $fields_per_type = $config->get('fields_per_type') ?: [];

    foreach ($options as $bundle => $label) {
      $form['content_types']["fields_$bundle"] = [
        '#type' => 'textfield',
        '#title' => $this->t('Fields to scan for @bundle', ['@bundle' => $label]),
        '#default_value' => isset($fields_per_type[$bundle]) ? implode(', ', $fields_per_type[$bundle]) : 'title, body, field_summary',
        '#description' => $this->t('Comma-separated list of field machine names to send to Pillarshield for this content type.'),
        '#states' => [
          'visible' => [
            // IMPORTANT: enabled_content_types is nested under content_types.
            ':input[name="content_types[enabled_content_types][' . $bundle . ']"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  
    public function testConnectionSubmit(array &$form, FormStateInterface $form_state): void {
  $config = $this->config('pillarshield.settings');

  // Resolve endpoint + key id from (1) nested, (2) flat, (3) saved config.
  $endpoint = $form_state->getValue(['api', 'api_endpoint']);
  if (!is_string($endpoint) || $endpoint === '') {
    $endpoint = $form_state->getValue('api_endpoint');
  }
  if (!is_string($endpoint) || $endpoint === '') {
    $endpoint = $config->get('api_endpoint');
  }
  $endpoint = is_string($endpoint) ? trim($endpoint) : '';

  $key_id = $form_state->getValue(['api', 'api_key_id']);
  if (!is_string($key_id) || $key_id === '') {
    $key_id = $form_state->getValue('api_key_id');
  }
  if (!is_string($key_id) || $key_id === '') {
    $key_id = $config->get('api_key_id');
  }
  $key_id = is_string($key_id) ? trim($key_id) : '';

  if ($endpoint === '' || $key_id === '') {
    $this->messenger()->addError($this->t('Missing endpoint or API key selection.'));
    return;
  }

  /** @var \Drupal\key\KeyRepositoryInterface $key_repo */
  $key_repo = \Drupal::service('key.repository');
  $key = $key_repo->getKey($key_id);

  if (!$key) {
    $this->messenger()->addError($this->t('Key entity could not be loaded (@id).', ['@id' => $key_id]));
    return;
  }

  $api_key = (string) $key->getKeyValue();
  if (trim($api_key) === '') {
    $this->messenger()->addError($this->t('Key value is empty (@id).', ['@id' => $key_id]));
    return;
  }

  try {
    /** @var \GuzzleHttp\ClientInterface $http */
    $http = \Drupal::httpClient();

    // Use "context" so audit log page_type/page_title are populated.
    $payload = [
      'api_key' => $api_key,
      'content' => '<p>PillarShield connector test.</p>',
      'context' => [
        'page_type' => 'settings_test',
        'page_title' => 'settings_test',
        'content_context' => 'ui:settings_test',
      ],
    ];

    $resp = $http->request('POST', $endpoint, [
      'headers' => [
        'Accept' => 'application/json',
        'Content-Type' => 'application/json',
      ],
      'json' => $payload,
      'timeout' => 10,
      'http_errors' => false,
    ]);

    $status = (int) $resp->getStatusCode();
    $raw = (string) $resp->getBody();
    $body = json_decode($raw, true);

    // Success criteria: JSON + has is_compliant.
    if ($status === 200 && is_array($body) && array_key_exists('is_compliant', $body)) {
      $rid = isset($body['request_id']) ? (string) $body['request_id'] : '';
      $msg = $this->t('OK. Endpoint reachable, key accepted, and response format is valid. HTTP @http.', ['@http' => $status]);
      if ($rid !== '') {
        $msg .= ' ' . $this->t('Request ID: @rid', ['@rid' => $rid]);
      }
      $this->messenger()->addStatus($msg);
      return;
    }

    // Helpful failure messages for the "status 200 but unexpected response" scenario.
    if ($raw !== '' && ($body === null) && json_last_error() !== JSON_ERROR_NONE) {
      $snippet = mb_substr($raw, 0, 600, 'UTF-8');
      $this->messenger()->addError($this->t('API returned non-JSON (or invalid JSON). HTTP @http. Snippet: @snippet', [
        '@http' => $status,
        '@snippet' => $snippet,
      ]));
      return;
    }

    if (!is_array($body)) {
      $snippet = mb_substr($raw, 0, 600, 'UTF-8');
      $this->messenger()->addError($this->t('Unexpected service response. HTTP @http. Snippet: @snippet', [
        '@http' => $status,
        '@snippet' => $snippet,
      ]));
      return;
    }

    // JSON but missing expected keys.
    $keys = implode(', ', array_slice(array_keys($body), 0, 20));
    $this->messenger()->addError($this->t('API returned JSON but not the expected shape (missing is_compliant). HTTP @http. Keys: @keys', [
      '@http' => $status,
      '@keys' => $keys,
    ]));
    return;

  }
  catch (\Throwable $e) {
    $this->messenger()->addError($this->t('Test request failed: @msg', ['@msg' => $e->getMessage()]));
    return;
  }
}


  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\Core\Config\Config $config */
    $config = $this->config('pillarshield.settings');

    // Option 1: values are nested under api/governance/content_types.
    $config->set('api_endpoint', (string) $form_state->getValue(['api', 'api_endpoint']));
    $config->set('api_key_id', (string) $form_state->getValue(['api', 'api_key_id']));
    $config->set('enable_governance', (bool) $form_state->getValue(['governance', 'enable_governance']));
    $config->set('allow_save_without_api', (bool) $form_state->getValue(['governance', 'allow_save_without_api']));

    $enabled_content_types = array_filter($form_state->getValue(['content_types', 'enabled_content_types']) ?: []);
    $config->set('enabled_content_types', $enabled_content_types);

    // Parse fields per type.
    $fields_per_type = [];
    foreach ($enabled_content_types as $bundle) {
      $raw = (string) $form_state->getValue(['content_types', "fields_$bundle"]);
      $fields = array_filter(array_map('trim', explode(',', $raw)));
      if ($fields) {
        $fields_per_type[$bundle] = $fields;
      }
    }
    $config->set('fields_per_type', $fields_per_type);

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
