<?php

declare(strict_types=1);

namespace Drupal\pinto\EventListener;

use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\Core\Url;
use Drupal\pinto\Controller\PintoAssetController;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

final class PintoRouteSubscriber extends RouteSubscriberBase {

  protected function alterRoutes(RouteCollection $collection): void {
    foreach (['css', 'js'] as $assetType) {
      // Suffix added as `.css` doesn't seem to be allowed by some nginx setups.
      $collection->add(\sprintf('pinto.asset.%s', $assetType), (new Route(path: \sprintf('/pinto/%s/{enumHash}/{enumCase}/{filePath}/workaround', $assetType)))
        ->setDefault('_controller', \sprintf('%s::%s', PintoAssetController::class, $assetType))
        ->addRequirements([
          '_access' => 'TRUE',
          // Regex of md5() (See PintoAssetController::hashEnum).
          'enumHash' => '^[a-f0-9]{32}$',
          // Set to max 32.
          'enumCase' => '^[A-Za-z_][A-Za-z0-9_]{1,32}$',
          'filePath' => '.+',
        ]),
      );
    }
  }

  /**
   * @phpstan-param 'css'|'js' $cssOrJs
   */
  public static function getAssetUrl(string $cssOrJs, string $enumHash, string $enumCase, string $path): Url {
    return Url::fromRoute(\sprintf('pinto.asset.%s', $cssOrJs), [
      'enumHash' => $enumHash,
      'enumCase' => $enumCase,
      'filePath' => $path,
    ]);
  }

}
