<?php

declare(strict_types=1);

namespace Drupal\pinto;

use Drupal\pinto\Object\PintoToDrupalBuilder;
use Drupal\pinto\ThemeDefinition\HookThemeDefinition;
use Pinto\Attribute\Build;
use Pinto\Attribute\Definition;
use Pinto\CanonicalProduct\Attribute\CanonicalProduct;
use Pinto\DefinitionDiscovery;
use Pinto\List\ObjectListInterface;
use Pinto\ObjectType\ObjectTypeDiscovery;
use Pinto\Slots\Definition as SlotsDefinition;
use Pinto\Slots\NoDefaultValue;
use Pinto\ThemeDefinition\HookThemeDefinition as PintoHookThemeDefinition;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Pinto compiler pass.
 *
 * Searches for Pinto enums in defined namespaces, then records the classnames
 * to the container so they may be efficiently fetched at runtime.
 */
final class PintoCompilerPass implements CompilerPassInterface {

  public function process(ContainerBuilder $container): void {
    /** @var array<class-string, string> $containerNamespaces */
    $containerNamespaces = $container->getParameter('container.namespaces');

    /** @var array<class-string<\Pinto\List\ObjectListInterface>> $pintoObjectLists */
    $pintoObjectLists = $container->getParameter('pinto.lists');

    /** @var string[] $pintoNamespaces */
    $pintoNamespaces = $container->getParameter('pinto.namespaces');

    $enumClasses = [];
    $enums = [];
    $definitions = [];
    $buildInvokers = [];
    $types = [];

    // Discover all known objects and their associated enum.
    $definitionDiscovery = new DefinitionDiscovery();
    foreach ($this->getEnums($containerNamespaces, $pintoNamespaces, $pintoObjectLists) as $r) {
      /** @var class-string<\Pinto\List\ObjectListInterface> $objectListClassName */
      $objectListClassName = $r->getName();
      $enumClasses[] = $objectListClassName;

      foreach ($r->getReflectionConstants() as $constant) {
        /** @var array<\ReflectionAttribute<\Pinto\Attribute\Definition>> $attributes */
        $attributes = $constant->getAttributes(Definition::class);
        $definition = ($attributes[0] ?? NULL)?->newInstance();
        if ($definition === NULL) {
          continue;
        }

        if ($container->hasDefinition($definition->className)) {
          // Ignore when this is a service.
          continue;
        }

        /** @var \Pinto\List\ObjectListInterface $case */
        $case = $constant->getValue();
        $definitionDiscovery[$definition->className] = $case;
      }
    }

    foreach ($definitionDiscovery as $objectClassName => $case) {
      $themeObjectDefinition = ObjectTypeDiscovery::definitionForThemeObject($objectClassName, $case, $definitionDiscovery);
      $types[$objectClassName] = $themeObjectDefinition[0];
      $definitions[$objectClassName] = $themeObjectDefinition[1];
      $buildInvokers[$objectClassName] = Build::buildMethodForThemeObject($objectClassName);
      $enums[$objectClassName] = [$case::class, $case->name];
    }

    $container->getDefinition(PintoMappingFactory::class)
      // $enumClasses is a separate parameter since there may be zero $enums.
      ->setArgument('$enumClasses', $enumClasses)
      ->setArgument('$enums', $enums)
      ->setArgument('$definitions', \serialize($definitions))
      ->setArgument('$buildInvokers', $buildInvokers)
      ->setArgument('$types', $types)
      ->setArgument('$lsbFactoryCanonicalObjectClasses', CanonicalProduct::discoverCanonicalProductObjectClasses($definitionDiscovery));

    // Precompile hook_theme().
    // Workaround https://www.drupal.org/project/drupal/issues/3522410
    $container->setParameter('pinto.internal.hook_theme', \serialize(\array_merge(...\array_map(static function (string $enumClass) use ($definitionDiscovery) {
      /** @var class-string<\Pinto\List\ObjectListInterface> $enumClass */
      $definitions = [];
      $enumDefinitions = $enumClass::definitions($definitionDiscovery);
      foreach ($enumDefinitions as $case) {
        $definition = $enumDefinitions[$case];

        $definitions[$case->name()] = match (TRUE) {
          // ThemeDefinition allows objects to define a hook_theme in full.
          $definition instanceof HookThemeDefinition => $definition->definition,
          $definition instanceof PintoHookThemeDefinition => $definition->definition,
          // Slots are a simplified (and recommended) since most objects don't
          // care about Drupal hook_theme-isms.
          $definition instanceof SlotsDefinition => ((static function () use ($case, $definition): array {
            $hookTheme = [
              'variables' => [],
              'path' => $case->templateDirectory(),
              'template' => $case->templateName(),
            ];

            // The keys of these are mapped at runtime in
            // \Drupal\pinto\Object\PintoToDrupalBuilder::transform().
            foreach ($definition->slots as $slot) {
              $slotName = $definition->renameSlots?->renamesTo($slot->name) ?? $slot->name;

              // Use null for when a slot has no default value.
              $hookTheme['variables'][PintoToDrupalBuilder::unitEnumToHookThemeVariableName($slotName)] = $slot->defaultValue instanceof NoDefaultValue ? NULL : $slot->defaultValue;
            }

            return $hookTheme;
          })()),
          default => throw new \LogicException(\sprintf('Unhandled definition for %s', $case->name))
        };
      }
      return $definitions;
    }, $enumClasses))));
  }

  /**
   * Get enums for the provided namespaces.
   *
   * @param array<class-string, string> $namespaces
   *   An array of namespaces. Where keys are class strings and values are
   *   paths.
   * @param string[] $pintoNamespaces
   *   Pinto namespaces.
   * @param array<class-string<\Pinto\List\ObjectListInterface>> $pintoObjectLists
   *   Object lists defined from service container parameters.
   *
   * @return \Generator<\ReflectionClass<\Pinto\List\ObjectListInterface>>
   *   Generates class strings.
   *
   * @throws \ReflectionException
   */
  private function getEnums(array $namespaces, array $pintoNamespaces, array $pintoObjectLists): \Generator {
    $yieldReflectionOnValidObjectList = static function (string $objectListClassName) {
      /** @var class-string $objectListClassName */
      /** @var \ReflectionClass<\Pinto\List\ObjectListInterface> $reflectionClass */
      $reflectionClass = new \ReflectionClass($objectListClassName);
      if ($reflectionClass->isEnum() && $reflectionClass->implementsInterface(ObjectListInterface::class)) {
        yield $reflectionClass;
      }
    };

    foreach ($namespaces as $namespace => $dirs) {
      $dirs = (array) $dirs;
      foreach ($dirs as $dir) {
        foreach ($pintoNamespaces as $pintoNamespace) {
          $nsDir = $dir . '/' . \str_replace('\\', '/', $pintoNamespace);
          if (!\file_exists($nsDir)) {
            continue;
          }
          $namespace .= '\\' . $pintoNamespace;

          /** @var \RecursiveIteratorIterator<\RecursiveDirectoryIterator<\SplFileInfo>> $iterator */
          $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($nsDir, \FilesystemIterator::KEY_AS_PATHNAME | \FilesystemIterator::CURRENT_AS_FILEINFO | \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST);
          foreach ($iterator as $fileinfo) {
            \assert($fileinfo instanceof \SplFileInfo);
            if ('php' !== $fileinfo->getExtension()) {
              continue;
            }

            /** @var \RecursiveDirectoryIterator|null $subDir */
            $subDir = $iterator->getSubIterator();
            if (NULL === $subDir) {
              continue;
            }

            $subDir = $subDir->getSubPath();
            $subDir = $subDir !== '' ? \str_replace(DIRECTORY_SEPARATOR, '\\', $subDir) . '\\' : '';

            /** @var class-string<\Pinto\List\ObjectListInterface> $className */
            $className = $namespace . '\\' . $subDir . $fileinfo->getBasename('.php');
            yield from $yieldReflectionOnValidObjectList($className);
          }
        }
      }
    }

    foreach ($pintoObjectLists as $pintoObjectList) {
      yield from $yieldReflectionOnValidObjectList($pintoObjectList);
    }
  }

  /**
   * Converts absolute asset file paths to relative to Drupal root.
   */
  public static function drupalRootLibraryAssets(array $libraries): array {
    $newLibraries = [];

    foreach ($libraries as $libraryName => $library) {
      $newLibraries[$libraryName] = \array_diff_key($library, \array_flip(['js', 'css']));

      foreach (($library['css'] ?? []) as $componentName => $component) {
        foreach ($component as $absoluteFileName => $definition) {
          $localizedFileName = (\str_contains($absoluteFileName, '://') || \str_starts_with($absoluteFileName, '//'))
            ? $absoluteFileName
            : (\str_starts_with($absoluteFileName, \DRUPAL_ROOT)
              ? \substr($absoluteFileName, \strlen(\DRUPAL_ROOT))
              : throw new \LogicException(\sprintf('Asset must be absolute, and begin with `%s`, unless the asset is in a stream wrapper. Found `%s`.', \DRUPAL_ROOT, $absoluteFileName))
            );

          $newLibraries[$libraryName]['css'][$componentName][$localizedFileName] = $definition;
        }
      }

      foreach (($library['js'] ?? []) as $absoluteFileName => $definition) {
        $localizedFileName = (\str_contains($absoluteFileName, '://') || \str_starts_with($absoluteFileName, '//'))
          ? $absoluteFileName
          : (\str_starts_with($absoluteFileName, \DRUPAL_ROOT)
            ? \substr($absoluteFileName, \strlen(\DRUPAL_ROOT))
            : throw new \LogicException(\sprintf('Asset must be absolute, and begin with `%s`, unless the asset is in a stream wrapper. Found `%s`.', \DRUPAL_ROOT, $absoluteFileName))
          );

        $newLibraries[$libraryName]['js'][$localizedFileName] = $definition;
      }
    }

    return $newLibraries;
  }

}
