<?php

declare(strict_types=1);

namespace Drupal\pinto\Library;

use Pinto\Library\DependencyCollection;
use Pinto\Library\LibraryBuilder;
use Pinto\List\Resource\ObjectListEnumResource;
use Pinto\PintoMapping;
use Pinto\Resource\ResourceInterface;

final class DrupalLibraryBuilder {

  /**
   * Implements hook_library_info_build().
   *
   * @internal
   */
  public static function libraryInfoBuild(PintoMapping $pintoMapping): array {
    $libraries = [];

    foreach ($pintoMapping->getResources() as $resource) {
      $library = static::library(
        resource: $resource,
        dependencies: LibraryBuilder::solveDeps($resource, $pintoMapping),
      );

      // Define the library if there is at least one asset or dependency.
      if ([] !== $library) {
        $libraries[$resource->libraryName()] = $library;
      }
    }

    return static::drupalRootLibraryAssets($libraries);
  }

  /**
   * Builds a library definition for a single resource.
   *
   * Uses assets, base JS directory, and CSS base directory provided by the resource.
   *
   * Dependencies to other resources must be provided externally.
   *
   * @phpstan-return array<string, array{css?: array<string, array<string, array<mixed>>>, js?: array<string, array<mixed>>}>
   */
  public static function library(
    ResourceInterface $resource,
    DependencyCollection $dependencies,
  ): array {
    $library = [];

    foreach (LibraryBuilder::expandLibraryPaths($resource) as [$asset, $libraryPath]) {
      /** @var \Pinto\Attribute\Asset\JsAssetInterface|\Pinto\Attribute\Asset\CssAssetInterface $asset */
      /** @var (array{'path': string}&array<string, mixed>) $vars */
      $vars = \get_object_vars($asset);
      unset($vars['path']);
      static::nestedValueSet($library, $libraryPath, $vars);
    }

    foreach ($dependencies as $dependency) {
      if ($dependency instanceof ObjectListEnumResource) {
        $on = $dependency->pintoEnum->attachLibraries();
      }
      else {
        $on = [$dependency];
      }

      $library['dependencies'] = [
        ...($library['dependencies'] ?? []),
        ...$on,
      ];
    }

    return $library;
  }

  /**
   * @phpstan-param array<mixed> $array
   * @phpstan-param string[] $keys
   */
  private static function nestedValueSet(array &$array, array $keys, mixed $value): void {
    $current = &$array;
    foreach ($keys as $key) {
      if (!isset($current[$key]) || !\is_array($current[$key])) {
        $current[$key] = [];
      }
      $current = &$current[$key];
    }
    $current = $value;
  }

  /**
   * Converts absolute asset file paths to relative to Drupal root.
   */
  public static function drupalRootLibraryAssets(array $libraries): array {
    $newLibraries = [];

    foreach ($libraries as $libraryName => $library) {
      $newLibraries[$libraryName] = \array_diff_key($library, \array_flip(['js', 'css']));

      foreach (($library['css'] ?? []) as $componentName => $component) {
        foreach ($component as $absoluteFileName => $definition) {
          $localizedFileName = (\str_contains($absoluteFileName, '://') || \str_starts_with($absoluteFileName, '//'))
            ? $absoluteFileName
            : (\str_starts_with($absoluteFileName, \DRUPAL_ROOT)
              ? \substr($absoluteFileName, \strlen(\DRUPAL_ROOT))
              : throw new \LogicException(\sprintf('Asset must be absolute, and begin with `%s`, unless the asset is in a stream wrapper. Found `%s`.', \DRUPAL_ROOT, $absoluteFileName))
            );

          $newLibraries[$libraryName]['css'][$componentName][$localizedFileName] = $definition;
        }
      }

      foreach (($library['js'] ?? []) as $absoluteFileName => $definition) {
        $localizedFileName = (\str_contains($absoluteFileName, '://') || \str_starts_with($absoluteFileName, '//'))
          ? $absoluteFileName
          : (\str_starts_with($absoluteFileName, \DRUPAL_ROOT)
            ? \substr($absoluteFileName, \strlen(\DRUPAL_ROOT))
            : throw new \LogicException(\sprintf('Asset must be absolute, and begin with `%s`, unless the asset is in a stream wrapper. Found `%s`.', \DRUPAL_ROOT, $absoluteFileName))
          );

        $newLibraries[$libraryName]['js'][$localizedFileName] = $definition;
      }
    }

    return $newLibraries;
  }

}
