<?php

declare(strict_types=1);

namespace Drupal\pinto\Object;

use Drupal\Core\Cache;
use Pinto\Slots;

/**
 * A slightly more opinionated trait, but improves DX, as an alternative to DrupalObjectTrait.
 *
 * If you need more extension points, you are encouraged to either use DrupalObjectTrait or copy-and-adapt this trait to your needs.
 *
 * Requirements:
 *   - This trait requires the object to use the _Slots_ object type.
 *   - In your object, simply implement the build() method.
 *
 * There is a higher chance of this trait being less performant, and providing features that may break with future core/Pinto versions.
 */
trait DrupalInvokableSlotsTrait {

  use DrupalObjectTrait;
  use Cache\RefinableCacheableDependencyTrait;

  final public function __invoke(): array {
    $built = $this->pintoBuild(function (mixed $build): mixed {
      \assert($build instanceof Slots\Build);
      return $this->build($build);
    });

    if (FALSE === \is_array($built)) {
      throw new \LogicException(\sprintf('Expected %s::transform to convert %s to an array.', PintoToDrupalBuilder::class, Slots\Build::class));
    }

    if ($this instanceof Cache\RefinableCacheableDependencyInterface) {
      (new Cache\CacheableMetadata())->addCacheableDependency($this)->applyTo($built);
    }

    return $built;
  }

  protected function build(Slots\Build $build): Slots\Build {
    // Objects will typically override this method.
    // In many cases, no build() method is necessary when `bindPromotedProperties` is used.
    return $build;
  }

}
