<?php

declare(strict_types=1);

namespace Drupal\pinto\Resource;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\pinto\List\SingleDirectoryObjectListTrait;
use Pinto\Attribute\Asset\AssetInterface;
use Pinto\Attribute\Asset\Css;
use Pinto\Attribute\Asset\Js;
use Pinto\Attribute\DependencyOn;
use Pinto\Resource\ResourceInterface;

/**
 * @internal
 */
final class SingleDirectoryObjectResource implements ResourceInterface {

  use SingleDirectoryObjectListTrait;

  /**
   * @phpstan-param class-string<object> $objectClassName
   */
  private function __construct(
    private readonly string $objectClassName,
  ) {
  }

  /**
   * @phpstan-param class-string<object> $objectClassName
   */
  public static function createFromClassName(string $objectClassName): static {
    return new static($objectClassName);
  }

  // @phpstan-ignore-next-line return.unusedType
  public function getClass(): ?string {
    return $this->objectClassName;
  }

  public function name(): string {
    // These standalone items don't get the opportunity to specify a name.
    return ContainerBuilder::hash($this->objectClassName);
  }

  public function libraryName(): string {
    return $this->name();
  }

  public function attachLibraries(): array {
    return [
      \sprintf('pinto/%s', $this->libraryName()),
    ];
  }

  public function build(callable $wrapper, object $object): callable {
    // Pass through.
    return $wrapper;
  }

  public function assets(): iterable {
    // Only the class can have assets.
    /** @var array<\ReflectionAttribute<AssetInterface>> $assets */
    $assets = (new \ReflectionClass($this->objectClassName))
      ->getAttributes(AssetInterface::class, \ReflectionAttribute::IS_INSTANCEOF);

    if ($assets !== []) {
      return \array_map(static fn (\ReflectionAttribute $r) => $r->newInstance(), $assets);
    }

    // If none then all in dir via glob.
    return [
      new Css(preprocess: TRUE),
      new Js(preprocess: TRUE),
    ];
  }

  public function dependencies(): iterable {
    return \array_map(static fn (\ReflectionAttribute $r) => $r->newInstance(), [
      // Above this class.
      ...(new \ReflectionClass($this->objectClassName))->getAttributes(DependencyOn::class),
    ]);
  }

}
