<?php

declare(strict_types=1);

namespace Drupal\Tests\pinto\Functional;

use Drupal\Core\Url;
use Drupal\pinto\Controller\PintoAssetController;
use Drupal\pinto\EventListener\PintoRouteSubscriber;
use Drupal\pinto_test\Pinto\Slots\Slots;
use Drupal\pinto_test\Pinto\StreamWrapper\StreamWrapper;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests Pinto rendering.
 *
 * @group pinto
 */
final class PintoTest extends BrowserTestBase {

  protected $defaultTheme = 'stark';

  protected static $modules = [
    'pinto_test_routes',
    'pinto_test',
    'pinto',
  ];

  /**
   * Test direct invocation of a theme object.
   *
   * @see \Drupal\pinto_test_routes\Controller\ObjectTestController::__invoke
   * @see \Drupal\pinto_test\Pinto\Generic\Generic
   * @see \Drupal\pinto_test\Pinto\Generic\Objects
   */
  public function testDirectInvocation(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.object_test'));
    $this->assertSession()->responseContains('Test text is foo bar!.');
    $this->assertSession()->responseContains('pinto_test/css/styles.css');
    $this->assertSession()->responseContains('pinto_test/js/app.js');
  }

  /**
   * Test directory based.
   *
   * @see \Drupal\pinto_test_routes\Controller\ObjectDirBased
   * @see \Drupal\pinto_test\Pinto\DirectoryBased\ObjectDirBased
   * @see \Drupal\pinto_test\Pinto\DirectoryBased\DirectoryBased
   */
  public function testDirBased(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.dir_based'));
    $this->assertSession()->responseContains('Dir based template for foo bar!.');
    $this->assertSession()->responseContains('pinto_test/resources/Object_Dir_Based/styles.css');
    $this->assertSession()->responseContains('pinto_test/resources/Object_Dir_Based/app.js');
  }

  /**
   * Test directory based on theme object location.
   *
   * @see \Drupal\pinto_test_routes\Controller\DirBasedWithPhpController::__invoke
   * @see \PintoResources\pinto_test\Object_Dir_Based_With_Php\ThemeObject
   * @see \Drupal\pinto_test\Pinto\DirectoryBasedWithPhp\DirectoryBasedWithPhp
   */
  public function testDirBasedOnThemeObjectLocation(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.dir_based_object_location'));
    $this->assertSession()->responseContains('Dir based template w/ PHP for foo bar!.');
    $this->assertSession()->responseContains('pinto_test/resources/Object_Dir_Based_With_Php/app.js');
    $this->assertSession()->responseContains('pinto_test/resources/Object_Dir_Based_With_Php/styles.css');
  }

  /**
   * Test nested rendering where inner object is predefined in the outer object.
   *
   * @see \Drupal\pinto_test_routes\Controller\NestedPredefinedController::__invoke
   * @see \Drupal\pinto_test\Pinto\Nested\ObjectNested
   * @see \Drupal\pinto_test\Pinto\Nested\ObjectNestedInner
   * @see \Drupal\pinto_test\Pinto\Nested\NestedObjects
   */
  public function testNestedPredefined(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.nested_predefined'));
    $this->assertSession()->responseContains(<<<FRAG
      Start outer inner [<span>[Start Inner [Nested inner text!] End Inner]</span>
      ] End outer inner.
      FRAG);
  }

  /**
   * Test nested rendering where inner object is passed into the outer object.
   *
   * @see \Drupal\pinto_test_routes\Controller\NestedObjectController::__invoke
   * @see \Drupal\pinto_test\Pinto\Nested\ObjectNestedViaVariable
   * @see \Drupal\pinto_test\Pinto\Nested\ObjectNestedInner
   * @see \Drupal\pinto_test\Pinto\Nested\NestedObjects
   */
  public function testNestedObject(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.nested_object'));
    $this->assertSession()->responseContains(<<<FRAG
      Start variable wrapped outer inner [<span>[Start Inner [Nested inner text in a nested variable!] End Inner]</span>
      ] End variable wrapped outer inner.
      FRAG);
  }

  /**
   * Tests slots.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsClassObjectController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectClass
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectClass
   */
  public function testSlotsObjectClass(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_class'));
    $this->assertSession()->pageTextContains('Text: Text! Number: 12345');
  }

  /**
   * Tests slots default values.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsClassObjectDefaultValueController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectDefaultValue
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectDefaultValue
   */
  public function testSlotsObjectDefaultValue(): void {
    /** @var string $hookTheme */
    $hookTheme = \Drupal::getContainer()->getParameter('pinto.internal.hook_theme');
    /** @var array<string, mixed> $hookTheme */
    $hookTheme = \unserialize($hookTheme);
    static::assertEquals([
      'variables' => [
        'text' => NULL,
        'number' => 333,
      ],
      'path' => '@pinto_test/templates/slots/',
      'template' => 'slots-object-default-value',
    ], $hookTheme[Slots::SlotsObjectDefaultValue->name]);
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_default_value'));
    $this->assertSession()->pageTextContains('Text: Text! Number: 333');
  }

  /**
   * Tests slots defined in the attribute.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectExplicitController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectExplicit
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectExplicit
   */
  public function testSlotsObjectExplicit(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_explicit'));
    $this->assertSession()->pageTextContains('Some text | 12345');
  }

  /**
   * Tests slots defined in the attribute, where slots are enums.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectExplicitEnumsController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectExplicitEnums
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectExplicitEnums
   */
  public function testSlotsObjectExplicitEnums(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_explicit_enums'));
    $this->assertSession()->pageTextContains('Slot One | 23456');
  }

  /**
   * Tests slots defined in the attribute, where slots are enums.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectBindPromotedPublicController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectBindPromotedPublic
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectBindPromotedPublic
   */
  public function testSlotsObjectBindPromotedPublic(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_bind_promoted_public'));
    $this->assertSession()->pageTextContains('Public text!');
  }

  /**
   * Tests slots defined in the attribute, where slots are enum class-strings.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectExplicitEnumClassController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectExplicitEnumClass
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectExplicitEnumClass
   */
  public function testSlotsObjectExplicitEnumClass(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_explicit_enum_class'));
    $this->assertSession()->pageTextContains('Slot One | Slot Two | Slot Three');
  }

  /**
   * Tests slots inherited from a #[Slots] object type attribute on a parent.
   *
   * Since no attribute is on an object method or object, Pinto will try
   * looking for the slot on a parent class, before enum case or enum.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectInheritanceController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectInheritance
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectInheritanceChild
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectInheritance
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectInheritanceChild
   */
  public function testSlotsObjectInheritance(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_inheritance'));
    $this->assertSession()->pageTextContains('Text slot!');
  }

  /**
   * Tests inherited slots with added slots.
   *
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectInheritanceModifySlotsController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectInheritanceModifySlotsAddSlots
   */
  public function testSlotsObjectInheritanceModifySlot(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_inheritance_modify_slot'));
    $this->assertSession()->pageTextContains('slotOnRoot: value for root');
    $this->assertSession()->pageTextContains('addedSlotsOnChild: value for added slot');
  }

  /**
   * Tests inherited slots with renamed Twig variable name.
   *
   * @covers \Drupal\pinto\Object\PintoToDrupalBuilder::transform
   * @covers \Drupal\pinto\PintoCompilerPass::process
   * @see \Drupal\pinto_test_routes\Controller\SlotsObjectInheritanceRenameSlotController::__invoke
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectInheritanceRenameSlot
   * @see \Drupal\pinto_test\Pinto\Slots\SlotsObjectInheritanceRenameSlotChild
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectInheritanceRenameSlot
   * @see \Drupal\pinto_test\Pinto\Slots\Slots::SlotsObjectInheritanceRenameSlotChild
   */
  public function testSlotsObjectInheritanceRenamedSlot(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.slots.object_inheritance_rename_slot'));
    $this->assertSession()->pageTextContains('Renamed slot!');
  }

  /**
   * Tests inherited slots with renamed Twig variable name.
   *
   * @see \Drupal\pinto_test_routes\Controller\CanonicalProductController::__invoke
   * @see \Drupal\pinto_test\Pinto\CanonicalProduct\CanonicalProductRoot
   * @see \Drupal\pinto_test\Pinto\CanonicalProduct\CanonicalProductExtending
   * @see \Drupal\pinto_test\Pinto\CanonicalProduct\CanonicalProductRootList
   * @see \Drupal\pinto_test\Pinto\CanonicalProduct\CanonicalProductExtendingList
   */
  public function testCanonicalProduct(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.canonical_product'));
    $this->assertSession()->pageTextContains('Value from Child! Text from controller!');
  }

  /**
   * Test glob assets.
   *
   * Only `styles.css` are included, but not `ignored-styles.css`.
   *
   * @see \Drupal\pinto_test\Pinto\Asset\GlobAsset
   */
  public function testGlobAssets(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.glob'));
    $this->assertSession()->responseContains('Text from AssetTestObject!');
    $this->assertSession()->responseContains('pinto_test/css/styles.css');
    $this->assertSession()->responseContains('pinto_test/css/styles-more.css');
    $this->assertSession()->responseNotContains('pinto_test/css/ignored-styles.css');
  }

  /**
   * Test dependency on parent object.
   *
   * Markup contains child values, assets are from parent and child.
   *
   * @see \Drupal\pinto_test\Pinto\Asset\DependencyOnAssetChild
   * @see \Drupal\pinto_test\Pinto\Asset\DependencyOnAssetChildObject
   * @see \Drupal\pinto_test\Pinto\Asset\DependencyOnAssetParent
   * @see \Drupal\pinto_test\Pinto\Asset\DependencyOnAssetParentObject
   * @see \Drupal\pinto_test_routes\Controller\DependencyOnAssetChild
   */
  public function testDependencyOnParentAssets(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.dependency_on.child'));
    $this->assertSession()->responseContains('DependencyOnAssetChildObject: Text from DependencyOnAssetChildObject!');
    $this->assertSession()->responseNotContains('Text from DependencyOnAssetParentObject!');
    $this->assertSession()->responseContains('pinto_test/css/child-styles.css');
    $this->assertSession()->responseContains('pinto_test/css/parent-styles.css');
  }

  /**
   * Test stream wrapper.
   *
   * @covers \Drupal\pinto\EventListener\PintoRouteSubscriber
   * @covers \Drupal\pinto\Controller\PintoAssetController
   * @covers \Drupal\pinto\StreamWrapper\PintoComponentStreamWrapper
   * @see \Drupal\pinto\List\StreamWrapperAssetTrait
   * @see \Drupal\pinto\List\StreamWrapperAssetInterface
   */
  public function testStreamWrapper(): void {
    $path = StreamWrapper::StreamWrapper->cssDirectory() . '/styles.css';
    static::assertEquals('pinto-component://css/9171db9908848033e039cde59f16387b/StreamWrapper/styles.css', $path);
    static::assertTrue(\file_exists($path));
    static::assertStringContainsString('border: 2px solid green;', \Safe\file_get_contents($path));

    $this->drupalGet(PintoRouteSubscriber::getAssetUrl(
      'css',
      PintoAssetController::hashEnum(StreamWrapper::class),
      StreamWrapper::StreamWrapper->name,
      'styles.css',
    ));
    $this->assertSession()->responseContains('border: 2px solid green;');
  }

  /**
   * Test standalone object.
   *
   * @see \Drupal\pinto_test_routes\Controller\StandaloneController::__invoke
   * @see \Drupal\pinto_test\Pinto\Standalone\StandaloneDefault\Standalone
   * @see \Drupal\pinto\Resource\SingleDirectoryObjectResource
   */
  public function testStandalone(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.standalone'));
    $this->assertSession()->responseContains('Standalone!');
    $this->assertSession()->responseContains('Text: Text!');
    $this->assertSession()->responseContains('Number: 12345');
    $this->assertSession()->responseContains('standalone-js.js');
    $this->assertSession()->responseContains('standalone-css.css');

    // And #[DependencyOn(PintoListDependencies::Charlie)]'s assets.
    $this->assertSession()->responseContains('pinto_test/js/app.js');
  }

  /**
   * Test standalone object.
   *
   * @see \Drupal\pinto_test_routes\Controller\StandaloneGlobController::__invoke
   * @see \Drupal\pinto_test\Pinto\Standalone\StandaloneGlob\StandaloneGlob
   * @see \Drupal\pinto\Resource\SingleDirectoryObjectResource
   */
  public function testStandaloneGlob(): void {
    $this->drupalGet(Url::fromRoute('pinto_test.standalone.glob'));
    $this->assertSession()->responseContains('StandaloneGlob!');
    $this->assertSession()->responseContains('Text: Text!');
    $this->assertSession()->responseNotContains('standalone-js.js');
    $this->assertSession()->responseContains('standalone-css.css');
  }

}
