<?php

declare(strict_types=1);

namespace Drupal\pinto\Build;

use Drupal\Core\Template\TwigEnvironment;
use Drupal\pinto\Library\DrupalLibraryBuilder;
use Drupal\pinto\Object\PintoToDrupalBuilder;
use Pinto\PintoMapping;
use Pinto\Resource\ResourceInterface;
use Pinto\Slots\Build;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * @todo allow preventing a build from an invokable from being rendered multiple times.
 * @internal
 * @see \Drupal\pinto\Element\PintoComponentElement
 */
final class BuildRegistry {

  /**
   * @phpstan-param \WeakMap<\Symfony\Component\HttpFoundation\Request, \WeakMap<\Drupal\pinto\Build\BuildToken, \Drupal\pinto\Build\BuildData>> $map
   */
  public function __construct(
    private readonly RequestStack $requestStack,
    private readonly PintoMapping $pintoMapping,
    private readonly TwigEnvironment $twigEnvironment,
    // @phpstan-ignore-next-line parameter.defaultValue
    private $map = new \WeakMap(),
  ) {
  }

  public function createToken(
    ResourceInterface $resource,
    Build $built,
  ): BuildToken {
    $currentRequest = $this->requestStack->getCurrentRequest() ?? throw new \Exception('Cannot build and create ' . BuildToken::class . ' when not in a request.');

    // @phpstan-ignore-next-line offsetAssign.valueType
    $map = $this->map[$currentRequest] ??= new \WeakMap();
    $token = BuildToken::createToken();
    $map[$token] = BuildData::createToken($resource, $built);
    return $token;
  }

  public function render(BuildToken $buildToken): array {
    $currentRequest = $this->requestStack->getCurrentRequest() ?? throw new \Exception('Cannot build and create ' . BuildToken::class . ' when not in a request.');
    $buildData = $this->map[$currentRequest][$buildToken] ?? throw new \Exception('Cannot find build data for provided token.');

    $objectClassName = $buildData->resource->getClass() ?? throw new \LogicException('Missing definition for slot');
    $definition = $this->pintoMapping->getThemeDefinition($objectClassName);
    if (!$definition instanceof \Pinto\Slots\Definition) {
      throw new \LogicException('Only slots type is supported at this time.');
    }

    /** @var array<string, mixed> $twigContext */
    $twigContext = [];
    foreach ($definition->slots as $slot) {
      $slotName = $definition->renameSlots?->renamesTo($slot->name) ?? $slot->name;
      $twigContext[PintoToDrupalBuilder::unitEnumToHookThemeVariableName($slotName)] = $buildData->built->pintoGet($slot->name);
    }

    return [
      '#markup' => $this->twigEnvironment->renderInline(
        template_string: \sprintf(<<<TWIG
          {%% include '%s' %%}
          TWIG,
          \sprintf('%s/%s.html.twig', $buildData->resource->templateDirectory(), $buildData->resource->templateName()),
        ),
        context: $twigContext,
      ),
      '#attached' => ['library' => DrupalLibraryBuilder::attachLibraries($buildData->resource)],
    ];
  }

}
