<?php

declare(strict_types=1);

namespace Drupal\pinto\Attribute\ObjectType;

use Drupal\pinto\HookTheme\Exception\PintoBuildDefinitionMismatch;
use Drupal\pinto\HookTheme\Exception\PintoThemeDefinition;
use Drupal\pinto\HookTheme\HookThemeDefinition;
use Drupal\pinto\Library\DrupalLibraryBuilder;
use Pinto\ObjectType\LateBindObjectContext;
use Pinto\ObjectType\ObjectTypeInterface;
use Pinto\Resource\ResourceInterface;

/**
 * An attribute representing the theme definition.
 *
 * - When attached to a class, the value of $definition must be set.
 * - When attached to a method, the $definition must not be set, the definition
 *   is instead returned by the method.
 *
 * The definition is merged into the default hook_theme definition.
 *
 * There is no need to define `template` or `path` key, as these are provided.
 * Though you may choose to override.
 *
 * This class replaces \Pinto\Attribute\ThemeDefinition.
 */
#[\Attribute(flags: \Attribute::TARGET_CLASS | \Attribute::TARGET_METHOD)]
final class ThemeDefinition implements ObjectTypeInterface {

  /**
   * Defines a theme definition.
   *
   * @param array<mixed>|null $definition
   *   Definition required when attached to a class. Otherwise, must be NULL.
   */
  final public function __construct(
    public ?array $definition = NULL,
  ) {
  }

  final public static function createBuild(ResourceInterface $resource, mixed $definition, string $objectClassName): mixed {
    return [
      '#theme' => $resource->name(),
      '#attached' => ['library' => DrupalLibraryBuilder::attachLibraries($resource)],
    ];
  }

  public static function lateBindObjectToBuild(mixed $build, mixed $definition, object $object, LateBindObjectContext $context): void {
  }

  final public static function validateBuild(mixed $build, mixed $definition, string $objectClassName): void {
    /* @phpstan-assert array{variables?: array<mixed>, path: string, template: string} $definition */
    if (FALSE === \is_array($build)) {
      // Allow build to be something other than an array.
      return;
    }

    if (!$definition instanceof HookThemeDefinition) {
      // Impossible, but for Stan.
      throw new \LogicException('Definition should be a ' . HookThemeDefinition::class);
    }

    $themeDefinitionKeysForComparison = \array_map(static fn (string $varName): string => '#' . $varName, \array_keys($definition->definition['variables'] ?? []));

    // @todo assert keys in $built map those in themeDefinition()
    // allow extra keys ( things like # cache).
    // But don't allow missing keys.
    $missingKeys = \array_diff($themeDefinitionKeysForComparison, \array_keys($build));
    if (\count($missingKeys) > 0) {
      throw new PintoBuildDefinitionMismatch($objectClassName, $missingKeys);
    }
  }

  final public function getDefinition(ResourceInterface $resource, \Reflector $r): mixed {
    if ($r instanceof \ReflectionClass) {
      $definition = $this->definition ?? throw new PintoThemeDefinition('$definition property must be set for ' . ObjectTypeInterface::class . ' attributes on the class level of a theme object.');
    }
    elseif ($r instanceof \ReflectionMethod) {
      if (NULL !== $this->definition) {
        throw new PintoThemeDefinition(\sprintf('%s attribute must not have $definition set on %s::%s.', ObjectTypeInterface::class, $r->getDeclaringClass()->getName(), $r->getName()));
      }

      if (FALSE === $r->isStatic()) {
        throw new PintoThemeDefinition(\sprintf('%s attribute must be attached to a static method. %s::%s is not static.', ObjectTypeInterface::class, $r->getDeclaringClass()->getName(), $r->getName()));
      }

      // Call the method directly and get the theme definition.
      /** @var array<mixed> $definition */
      $definition = $r->invoke(NULL);
    }

    // @phpstan-ignore-next-line argument.type
    return new HookThemeDefinition(definition: ($definition ?? []) + [
      'variables' => [],
      'path' => $resource->templateDirectory(),
      'template' => $resource->templateName(),
    ]);
  }

}
