<?php

declare(strict_types=1);

namespace Drupal\pinto\List;

use Composer\Autoload\ClassLoader;

/**
 * @internal
 */
final class SingleDirectoryObjectListCache {

  /**
   * The nearest module root.
   *
   * Internal, may be removed at any time.
   *
   * Since some installations, like DrupalCI, may use symlinks, and __DIR__
   * always uses the resolved dir. So cant use the dir, just use the
   * "@-modulename" form for Twig. Alternatively if this doesn't work out, use
   * a custom \Twig\Loader\LoaderInterface.
   *
   * @internal
   */
  private static array $nsAndDir = [];

  /**
   * The app-relative directory of an object class.
   *
   * Internal, may be removed at any time.
   *
   * @internal
   */
  private static array $libraryDir = [];

  /**
   * @phpstan-param class-string $objectClassName
   */
  public static function twigNsAndDir(string $objectClassName): string {
    if (isset(static::$nsAndDir[$objectClassName])) {
      return static::$nsAndDir[$objectClassName];
    }

    /** @var non-empty-string $fileName */
    $fileName = (new \ReflectionClass($objectClassName))->getFileName();
    $objectClassDir = \dirname($fileName);
    $dir = $objectClassDir;
    while (TRUE) {
      /** @var \RecursiveIteratorIterator<\SplFileInfo> $iterator */
      // @phpstan-ignore-next-line generics.notSubtype
      $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($dir, \FilesystemIterator::KEY_AS_PATHNAME | \FilesystemIterator::CURRENT_AS_FILEINFO | \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST);
      /** @var \SplFileInfo $fileinfo */
      foreach ($iterator as $fileinfo) {
        if (\str_ends_with($fileinfo->getFilename(), '.info.yml')) {
          $moduleName = $fileinfo->getBasename('.info.yml');
          return (static::$nsAndDir[$objectClassName] = '@' . $moduleName . \substr($objectClassDir, \strlen($dir)));
        }
      }

      $dir = \dirname($dir);
    }
  }

  /**
   * @phpstan-param class-string $objectClassName
   */
  public static function libraryDir(string $objectClassName): string {
    if (isset(static::$libraryDir[$objectClassName])) {
      return static::$libraryDir[$objectClassName];
    }

    // The file might actually be in the Drupal install, but symlinked. Get the
    // real file location via ClassLoader.
    $fileName = static::classLoader()->findFile($objectClassName);
    if (FALSE === $fileName) {
      throw new \LogicException(\sprintf('Unable to get file for %s from class loader', $objectClassName));
    }

    $objectClassDir = \Safe\realpath(\dirname($fileName));
    if (\str_starts_with($objectClassDir, \DRUPAL_ROOT)) {
      return static::$libraryDir[$objectClassName] = $objectClassDir;
    }

    // This can happen on Drupal CI, or setups where module is symlinked
    // where code lies outside the Drupal install.
    throw new \LogicException(\sprintf('Somehow the class is not in the Drupal directory: %s is not in %s', $objectClassDir, \DRUPAL_ROOT));
  }

  private static function classLoader(): ClassLoader {
    return \Drupal::service('class_loader');
  }

}
