<?php

declare(strict_types=1);

namespace Drupal\Tests\pinto\Kernel;

use Drupal\Core\Render\RendererInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\pinto\Build\BuildRegistryInterface;
use Drupal\pinto\Build\BuildToken;
use Drupal\pinto\Element\PintoComponentElement;
use Drupal\pinto_test\Pinto\Slots\SlotsObjectClass;

/**
 * Tests Build registry system and element.
 *
 * @see \Drupal\pinto\Element\PintoComponentElement
 * @see \Drupal\pinto\Build\BuildToken
 * @see \Drupal\pinto\Build\BuildData
 * @see \Drupal\pinto\Build\BuildRegistry
 *
 * @phpstan-import-type RenderArray from \Drupal\pinto\Element\PintoComponentElement
 * @group pinto
 */
final class PintoBuildRegistryTest extends KernelTestBase {

  protected static $modules = [
    'pinto_test',
    'pinto',
  ];

  /**
   * Test rendering a component, and internals of BuildRegistry.
   *
   * BuildRegistry works in kernel tests since \Drupal\KernelTests\KernelTestBase::bootKernel pushes a Request into
   * the stack.
   */
  public function testSlotsRenderToComponent(): void {
    /**
     * @phpstan-return array<\Drupal\pinto\Build\BuildData>
     * @phpstan-impure
     */
    $getBuildData = static function (): array {
      $buildRegistry = \Drupal::service(BuildRegistryInterface::class);
      $r = new \ReflectionClass($buildRegistry);
      $p = $r->getProperty('map');
      $p->setAccessible(TRUE);
      /** @var \WeakMap<\Symfony\Component\HttpFoundation\Request, \WeakMap<\Drupal\pinto\Build\BuildToken, \Drupal\pinto\Build\BuildData>> $map */
      $map = $p->getValue($buildRegistry);
      $buildDatas = [];
      foreach ($map as $builds) {
        foreach ($builds as $buildData) {
          $buildDatas[] = $buildData;
        }
      }
      return $buildDatas;
    };

    $text = $this->randomMachineName();
    $obj = new SlotsObjectClass($text, 1234);
    /** @var RenderArray $built */
    $built = $obj();
    static::assertEquals(PintoComponentElement::class, $built['#type']);
    // @phpstan-ignore-next-line staticMethod.alreadyNarrowedType
    static::assertInstanceOf(BuildToken::class, $built['#buildToken']);
    static::assertCount(1, $getBuildData());
    $rendered = (string) static::renderer()->renderRoot($built);
    // $built is mutated, causing the ref to the BuildToken to go away, so the weakref in BuildRegistry releases.
    static::assertEquals(0, \count($getBuildData()));
    static::assertStringContainsString('Text: ' . $text, $rendered);
    static::assertStringContainsString('Number: 1234', $rendered);
  }

  /**
   * Tests if values like #attached are passed along with a component they are retained after going to BuildRegistry.
   *
   * This is necessary since if we replace Drupal's page component, we need to retain the values added via attached.
   */
  public function testRetainedValues(): void {
    $obj = new SlotsObjectClass('foo', 1234);
    $built = $obj();
    static::assertIsArray($built);
    // @phpstan-ignore-next-line offsetAccess.nonOffsetAccessible
    $built['#attached']['library'][] = 'fake library';
    static::renderer()->renderRoot($built);
    static::assertCount(2, $built['#attached']['library']);
    // One of those is our mutated library, the other is the one for SlotsObjectClass.
    static::assertContainsEquals('fake library', $built['#attached']['library']);
  }

  public static function renderer(): RendererInterface {
    return \Drupal::service(RendererInterface::class);
  }

}
