<?php

declare(strict_types=1);

namespace Drupal\pinto\Element;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\RenderElementBase;
use Drupal\pinto\Build\BuildRegistryInterface;
use Pinto\Resource\ResourceInterface;
use Pinto\Slots\Build;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Internal Pinto component element.
 *
 * This element must not be created directly by end user code. Renders arrays for PintoComponentElement must be
 * generated by Pinto code only, the internals may change at any time.
 *
 * When executing a non-HookTheme Pinto component, a PintoComponentElement render array will be returned with an
 * internally tracked build token.
 * A Pinto components build object will be tracked to BuildRegistry.
 * Once all render arrays containing references to the relevant build object are rendered, PHP cleans up Resource
 * and Build Object references automatically.
 * Thusly, a slightly higher memory usage is exchanged for deferred rendering to final render time, and potentially less
 * CPU and memory usage since some objects may only be conditionally rendered.
 *
 * Note: Render arrays are not serializable! A render array containing a reference to a PintoComponentElement must be rendered within the same request as
 * it's generated, since tracked build tokens and their related build objects expire at the end of a request.
 *
 * @phpstan-type RenderArray array{
 *   "#type": class-string<static>,
 *   "#buildToken": \Drupal\pinto\Build\BuildToken,
 * }
 * @see \Drupal\pinto\Build\BuildRegistry
 * @internal
 */
#[RenderElement(id: self::class)]
final class PintoComponentElement extends RenderElementBase implements ContainerFactoryPluginInterface {

  /**
   * @internal
   * @phpstan-ignore-next-line constructor.missingParentCall
   */
  public function __construct(
    private readonly BuildRegistryInterface $buildRegistry,
  ) {
    // No parent call: we don't need context of the Plugin, when it's a plugin.
  }

  /**
   * An alternative entry point when its created as a plugin.
   *
   * This might be able to be removed when AutowirePluginTrait is available on all supported core version.
   *
   * @internal
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $container->get(BuildRegistryInterface::class),
    );
  }

  /**
   * @phpstan-return array<string, mixed>
   */
  public function getInfo(): array {
    return [
      '#pre_render' => [
        // A callable for \Drupal\Core\Utility\CallableResolver::getCallableFromDefinition
        // Called in \Drupal\Core\Render\Renderer::doCallback
        // \mglaman\PHPStanDrupal\Rules\Drupal\RenderCallbackRule is wrong:
        \sprintf('%s:preRender', static::class),
      ],
    ];
  }

  /**
   * @phpstan-param RenderArray $element
   */
  public function preRender($element): array {
    ['#buildToken' => $buildToken] = $element;
    unset($element['#type']);
    unset($element['#buildToken']);
    return $this->buildRegistry->render($buildToken, $element);
  }

  /**
   * @phpstan-return RenderArray
   */
  public static function renderArray(Build $built, ResourceInterface $resource): array {
    return [
      '#type' => static::class,
      '#buildToken' => static::buildRegistry()->createToken($resource, $built),
    ];
  }

  private static function buildRegistry(): BuildRegistryInterface {
    /** @var \Drupal\pinto\Build\BuildRegistry */
    return \Drupal::service(BuildRegistryInterface::class);
  }

}
