<?php

declare(strict_types=1);

namespace Drupal\pinto;

use Drupal\pinto\HookTheme\HookThemeDefinition;
use Drupal\pinto\Resource\SingleDirectoryObjectResource;
use Pinto\Attribute\Build;
use Pinto\CanonicalProduct\Attribute\CanonicalProduct;
use Pinto\DefinitionDiscovery;
use Pinto\Exception\PintoObjectTypeDefinition;
use Pinto\List\ObjectListInterface;
use Pinto\List\Resource\ObjectListEnumResource;
use Pinto\ObjectType\ObjectTypeDiscovery;
use Pinto\Resource\ResourceInterface;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Pinto compiler pass.
 *
 * Searches for Pinto enums in defined namespaces, then records the classnames
 * to the container so they may be efficiently fetched at runtime.
 */
final class PintoCompilerPass implements CompilerPassInterface {

  public function process(ContainerBuilder $container): void {
    /** @var array<class-string, string> $containerNamespaces */
    $containerNamespaces = $container->getParameter('container.namespaces');

    /** @var array<class-string<\Pinto\List\ObjectListInterface>> $pintoObjectLists */
    $pintoObjectLists = $container->getParameter('pinto.lists');

    /** @var string[] $pintoNamespaces */
    $pintoNamespaces = $container->getParameter('pinto.namespaces');

    /** @var array<class-string, ResourceInterface> $resources */
    $resources = [];
    $definitions = [];
    /** @var \SplObjectStorage<\Pinto\Resource\ResourceInterface, mixed> $definitionsByResource */
    $definitionsByResource = new \SplObjectStorage();
    $buildInvokers = [];
    $types = [];

    /** @var array<class-string<object>> $discoveredComponents */
    // @phpstan-ignore-next-line
    $discoveredComponents = \array_values($container->getParameter('pinto.components'));

    // Discover all known objects and their associated enum.
    $definitionDiscovery = new DefinitionDiscovery();
    foreach ($this->discoverClasses($containerNamespaces, $pintoNamespaces, $pintoObjectLists) as [$componentOrEnum, $r]) {
      if ($componentOrEnum === 'component') {
        $discoveredComponents[] = $r->getName();

        continue;
      }

      foreach ($r->getReflectionConstants() as $constant) {
        /** @var \Pinto\List\ObjectListInterface $case */
        $case = $constant->getValue();

        $objectClassName = $case->getClass();
        if ($objectClassName === NULL) {
          // This enum case doesn't have a Definition attribute.
          // The ID is not a part of the API, and thusly this entry cannot be requested by PintoMapping::getResource().
          // However it can be iterated in PintoMapping::getResources().
          $resources[\sprintf('classless-%s', ContainerBuilder::hash($case))] = ObjectListEnumResource::createFromEnum($case);

          continue;
        }

        if ($container->hasDefinition($objectClassName)) {
          // Ignore when this is a service.
          continue;
        }

        $definitionDiscovery[$objectClassName] = ObjectListEnumResource::createFromEnum($case);
      }
    }

    // PHP classes not associated with a list might not all be components, so we just ignore when
    // definitionForThemeObject runs on it.
    $silentInvalidDiscovery = [];

    // Register standalone components.
    // Create resources for discovered components which are not a part of a list.
    foreach (\array_unique($discoveredComponents) as $className) {
      if (FALSE === isset($definitionDiscovery[$className])) {
        // We make everything a `SingleDirectoryObjectResource` for now...
        $definitionDiscovery[$className] = SingleDirectoryObjectResource::createFromClassName($className);
        $silentInvalidDiscovery[] = $className;
      }
    }

    foreach ($definitionDiscovery as $objectClassName => $resource) {
      try {
        $themeObjectDefinition = ObjectTypeDiscovery::definitionForThemeObject($objectClassName, $resource, $definitionDiscovery);
      }
      catch (PintoObjectTypeDefinition $e) {
        if (\in_array($objectClassName, $silentInvalidDiscovery, TRUE)) {
          continue;
        }
        else {
          throw $e;
        }
      }

      $types[$objectClassName] = $themeObjectDefinition[0];
      $definitionsByResource[$resource] = $definitions[$objectClassName] = $themeObjectDefinition[1];
      $buildInvokers[$objectClassName] = Build::buildMethodForThemeObject($objectClassName);
      $resources[$objectClassName] = $resource;
    }

    $container->getDefinition(PintoMappingFactory::class)
      ->setArgument('$resources', \serialize($resources))
      ->setArgument('$definitions', \serialize($definitions))
      ->setArgument('$buildInvokers', $buildInvokers)
      ->setArgument('$types', $types)
      ->setArgument('$lsbFactoryCanonicalObjectClasses', CanonicalProduct::discoverCanonicalProductObjectClasses($definitionDiscovery));

    // Precompile hook_theme().
    // Serialize to work around https://www.drupal.org/project/drupal/issues/3522410
    $container->setParameter('pinto.internal.hook_theme', \serialize($this::hookThemes($resources, static function (ResourceInterface $resource) use ($definitionsByResource): mixed {
        return $definitionsByResource->offsetExists($resource) ? $definitionsByResource[$resource] : NULL;
    })));
  }

  /**
   * Get enums for the provided namespaces.
   *
   * @param array<class-string, string> $namespaces
   *   An array of namespaces. Where keys are class strings and values are
   *   paths.
   * @param string[] $pintoNamespaces
   *   Pinto namespaces.
   * @param array<class-string<\Pinto\List\ObjectListInterface>> $pintoObjectLists
   *   Object lists defined from service container parameters.
   *
   * @return \Generator<
   *   array{'enum', \ReflectionClass<\Pinto\List\ObjectListInterface>}|array{'component', \ReflectionClass<\Pinto\List\ObjectListInterface>}
   *   >
   *   Reflections of enum classes.
   *
   * @throws \ReflectionException
   */
  private function discoverClasses(array $namespaces, array $pintoNamespaces, array $pintoObjectLists): \Generator {
    $yieldReflectionOnValidObjectList = static function (string $objectListClassName) {
      /** @var class-string $objectListClassName */
      /** @var \ReflectionClass<\Pinto\List\ObjectListInterface> $reflectionClass */
      $reflectionClass = new \ReflectionClass($objectListClassName);
      if ($reflectionClass->isEnum() && $reflectionClass->implementsInterface(ObjectListInterface::class)) {
        yield ['enum', $reflectionClass];
      }
      else {
        yield ['component', $reflectionClass];
      }
    };

    foreach ($namespaces as $namespace => $dirs) {
      $dirs = (array) $dirs;
      foreach ($dirs as $dir) {
        foreach ($pintoNamespaces as $pintoNamespace) {
          $nsDir = $dir . '/' . \str_replace('\\', '/', $pintoNamespace);
          if (!\file_exists($nsDir)) {
            continue;
          }

          /** @var \RecursiveIteratorIterator<\RecursiveDirectoryIterator<\SplFileInfo>> $iterator */
          $iterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($nsDir, \FilesystemIterator::KEY_AS_PATHNAME | \FilesystemIterator::CURRENT_AS_FILEINFO | \RecursiveDirectoryIterator::SKIP_DOTS), \RecursiveIteratorIterator::SELF_FIRST);
          foreach ($iterator as $fileinfo) {
            \assert($fileinfo instanceof \SplFileInfo);
            if ('php' !== $fileinfo->getExtension()) {
              continue;
            }

            /** @var \RecursiveDirectoryIterator<\SplFileInfo>|null $subDir */
            // @phpstan-ignore-next-line varTag.type
            $subDir = $iterator->getSubIterator();
            if (NULL === $subDir) {
              continue;
            }

            $subDir = $subDir->getSubPath();
            $subDir = $subDir !== '' ? \str_replace(DIRECTORY_SEPARATOR, '\\', $subDir) . '\\' : '';

            /** @var class-string<\Pinto\List\ObjectListInterface> $className */
            $className = \sprintf('%s\%s\%s%s', $namespace, $pintoNamespace, $subDir, $fileinfo->getBasename('.php'));
            yield from $yieldReflectionOnValidObjectList($className);
          }
        }
      }
    }

    foreach ($pintoObjectLists as $pintoObjectList) {
      yield from $yieldReflectionOnValidObjectList($pintoObjectList);
    }
  }

  public static function hookThemes(array $resources, callable $getDefinition): array {
    $hookThemeDefs = [];
    foreach ($resources as $resource) {
      $c = $resource->getClass();
      if ($c === NULL) {
        continue;
      }

      $definition = $getDefinition($resource) ?? NULL;
      if (!$definition instanceof HookThemeDefinition) {
        continue;
      }

      // ThemeDefinition defines a hook_theme in full.
      $hookThemeDefs[$resource->name()] = $definition->definition;
    }

    return $hookThemeDefs;
  }

}
