<?php

declare(strict_types=1);

namespace Drupal\pinto\StreamWrapper;

use Drupal\Core\StreamWrapper\LocalReadOnlyStream;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\pinto\Controller\PintoAssetController;
use Drupal\pinto\EventListener\PintoRouteSubscriber;
use Drupal\pinto\List\StreamWrapperAssetInterface;
use Pinto\Attribute\Asset\Css;
use Pinto\Attribute\Asset\Js;
use Pinto\List\Resource\ObjectListEnumResource;
use Pinto\PintoMapping;

final class PintoComponentStreamWrapper extends LocalReadOnlyStream {

  public function getName(): TranslatableMarkup {
    return \t('Pinto.');
  }

  public function getDescription(): TranslatableMarkup {
    return \t('Pinto.');
  }

  /**
   * {@inheritdoc}
   */
  public function getDirectoryPath() {
    [1 => $afterProto] = \explode('://', $this->uri, 2);
    [$cssOrJs, $enumHash, $enumCase, $path] = \explode('/', $afterProto, 4);

    foreach (static::getEnumClasses() as $enumClass) {
      if ($enumHash === PintoAssetController::hashEnum($enumClass)) {
        $enumStr = \sprintf('%s::%s', $enumClass, $enumCase);
        if (FALSE === \defined($enumStr)) {
          throw new \LogicException('Bad path!');
        }

        /** @var \Pinto\List\ObjectListInterface $enum */
        $enum = \constant($enumStr);
        if (!$enum instanceof StreamWrapperAssetInterface) {
          throw new \LogicException(\sprintf('Enum does not implement %s', StreamWrapperAssetInterface::class));
        }

        return match ($cssOrJs) {
          'css' => $enum->absoluteCssDirectory(),
          'js' => $enum->absoluteJsDirectory(),
          default => throw new \LogicException(),
        };
      }
    }

    throw new \LogicException('Bad path!');
  }

  protected function getTarget($uri = NULL): string {
    [1 => $afterProto] = \explode('://', $this->uri, 2);
    [$cssOrJs, $enumHash, $enumCase, $path] = \explode('/', $afterProto, 4);

    foreach (static::getEnumClasses() as $enumClass) {
      if ($enumHash === PintoAssetController::hashEnum($enumClass)) {
        $enumStr = \sprintf('%s::%s', $enumClass, $enumCase);
        if (FALSE === \defined($enumStr)) {
          throw new \LogicException('Bad path!');
        }

        /** @var \Pinto\List\ObjectListInterface $enum */
        $enum = \constant($enumStr);
        foreach ($enum->assets() as $asset) {
          // @todo change to JsAssetInterface and CssAssetInterface.
          if (($cssOrJs === 'js' && $asset instanceof Js) || ($cssOrJs === 'css' && $asset instanceof Css)) {
            // Validate path is one that exists on the enum.
            if ($asset->path === $path) {
              return $path;
            }
          }
        }
      }
    }

    throw new \LogicException('Bad path!');
  }

  /**
   * Override getExternalUrl().
   *
   * Return the HTML URI of a public file.
   */
  public function getExternalUrl(): string {
    [1 => $afterProto] = \explode('://', $this->uri, 2);
    /** @var 'css'|'js' $cssOrJs */
    [$cssOrJs, $enumHash, $enumCase, $path] = \explode('/', $afterProto, 4);
    return PintoRouteSubscriber::getAssetUrl(
      $cssOrJs,
      $enumHash,
      $enumCase,
      $path,
    )->setOption('path_processing', FALSE)->setAbsolute()->toString();
  }

  /**
   * Use a mapping pre-made in the container.
   *
   * @internal
   */
  private static function pintoMapping(): PintoMapping {
    /** @var \Pinto\PintoMapping */
    return \Drupal::service(PintoMapping::class);
  }

  /**
   * @phpstan-return array<class-string<\Pinto\List\ObjectListInterface>>
   */
  private static function getEnumClasses(): array {
    $enumClasses = [];
    foreach (static::pintoMapping()->getResources() as $resource) {
      if ($resource instanceof ObjectListEnumResource) {
        $enumClasses[] = $resource->pintoEnum::class;
      }
    }
    return \array_values(\array_unique($enumClasses));
  }

}
