<?php

declare(strict_types=1);

namespace Drupal\Tests\pinto\Unit;

use Drupal\pinto\Library\DrupalLibraryBuilder;
use Drupal\pinto_test\Pinto\DependencyOn\PintoListDependencies;
use Drupal\pinto_test\Pinto\DependencyOn\PintoListDependenciesHierarchyChild;
use Drupal\pinto_test\Pinto\DependencyOn\PintoListDependenciesHierarchyParent;
use Drupal\pinto_test\Pinto\DependencyOn\PintoObjectDependencyOnChild;
use Drupal\pinto_test\Pinto\DependencyOn\PintoObjectDependencyOnParent;
use PHPUnit\Framework\TestCase;
use Pinto\CanonicalProduct\Attribute\CanonicalProduct;
use Pinto\DefinitionDiscovery;
use Pinto\List\Resource\ObjectListEnumResource;
use function Safe\realpath;

/**
 * Tests pinto enums where cases do not have definitions.
 *
 * @see PintoListDependencies
 */
final class PintoDependenciesTest extends TestCase {

  /**
   * @covers \Pinto\List\ObjectListTrait::assets
   * @covers \Pinto\Attribute\DependencyOn
   * @covers \Drupal\pinto\Library\DrupalLibraryBuilder::drupalRootLibraryAssets
   */
  public function testAssets(): void {
    // Source paths from attributes, where drupalRootLibraryAssets chops them off:
    $rootLess = static function (string $absoluteFileName): string {
      return \substr($absoluteFileName, \strlen(\DRUPAL_ROOT));
    };

    $pintoMapping = new \Pinto\PintoMapping(
      resources: [
        '--test-1' => ObjectListEnumResource::createFromEnum(PintoListDependencies::Alpha),
        '--test-2' => ObjectListEnumResource::createFromEnum(PintoListDependencies::Beta),
        '--test-3' => ObjectListEnumResource::createFromEnum(PintoListDependencies::Charlie),
        '--test-4' => ObjectListEnumResource::createFromEnum(PintoListDependencies::Delta),
      ],
      definitions: [],
      buildInvokers: [],
      types: [],
      lsbFactoryCanonicalObjectClasses: [],
    );

    static::assertEquals([
      PintoListDependencies::Alpha->value => [
        'dependencies' => [
          'pinto/beta',
          'pinto/charlie',
        ],
      ],
      PintoListDependencies::Beta->value => [
        'js' => [
                // These paths must be normalized according to DrupalLibraryBuilder::drupalRootLibraryAssets().
          $rootLess(realpath(__DIR__ . '/../../modules/pinto_test/js/app.js')) => [
            'minified' => FALSE,
            'preprocess' => FALSE,
            'attributes' => [],
            'silenceNoMatches' => FALSE,
          ],
        ],
        'css' => [
          'component' => [
            $rootLess(realpath(__DIR__ . '/../../modules/pinto_test/css/styles.css')) => [
              'minified' => FALSE,
              'preprocess' => FALSE,
              'category' => 'component',
              'attributes' => [],
              'silenceNoMatches' => FALSE,
            ],
          ],
        ],
        'dependencies' => [
          'pinto/charlie',
        ],
      ],
      PintoListDependencies::Charlie->value => [
        'js' => [
          $rootLess(realpath(__DIR__ . '/../../modules/pinto_test/js/app.js')) => [
            'minified' => FALSE,
            'preprocess' => FALSE,
            'attributes' => [],
            'silenceNoMatches' => FALSE,
          ],
        ],
      ],
      PintoListDependencies::Delta->value => [
        'dependencies' => [
          'pinto/alpha',
          'foo/bar',
        ],
      ],
    ], DrupalLibraryBuilder::libraryInfoBuild($pintoMapping));
  }

  /**
   * Test DependencyOn(parent).
   */
  public function testDependencyOnParent(): void {
    $definitionDiscovery = new DefinitionDiscovery();
    $definitionDiscovery[PintoObjectDependencyOnChild::class] = PintoListDependenciesHierarchyChild::Child;
    $definitionDiscovery[PintoObjectDependencyOnParent::class] = PintoListDependenciesHierarchyParent::Parent;

    $pintoMapping = new \Pinto\PintoMapping(
          resources: [
            PintoObjectDependencyOnChild::class => ObjectListEnumResource::createFromEnum(PintoListDependenciesHierarchyChild::Child),
            PintoObjectDependencyOnParent::class => ObjectListEnumResource::createFromEnum(PintoListDependenciesHierarchyParent::Parent),
          ],
          definitions: [],
          buildInvokers: [],
          types: [],
          lsbFactoryCanonicalObjectClasses: CanonicalProduct::discoverCanonicalProductObjectClasses($definitionDiscovery),
      );
    static::assertEquals([
      'PintoListDependenciesHierarchyChild-dependency-on-Child' => [
        'dependencies' => [
          'pinto/PintoListDependenciesHierarchyParent-dependency-on-Parent',
        ],
      ],
    ], DrupalLibraryBuilder::libraryInfoBuild($pintoMapping));
  }

}
