<?php

declare(strict_types=1);

namespace Drupal\pinto_entity;

use Drupal\Core\Entity\EntityInterface;
use Drupal\pinto_entity\EntityView\Attribute\EntityView;
use Drupal\pinto_entity\EntityView\Bundle;
use Drupal\pinto_entity\EntityView\EntityViewContext;
use Pinto\PintoMapping;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * @phpstan-import-type PintoEntityMapping from \Drupal\pinto_entity\DependencyInjection\PintoEntityCompilerPass
 */
final class PintoEntity {

  /**
   * @var PintoEntityMapping
   */
  private array $componentMapping;

  /**
   * @var array<string, Bundle>
   */
  private $bundles = [];

  /**
   * @var \SplObjectStorage<
   *   Bundle,
   *   array<string, array{
   *     \Drupal\pinto_entity\EntityView\ViewMode,
   *     \ReflectionMethod,
   *     string[],
   *     string[],
   *   }>
   * >
   */
  private \SplObjectStorage $mappingByBundle;

  /**
   * PintoEntity service.
   *
   * Closure is used to work around circular references.
   *
   * @phpstan-param (\Closure(): \Drupal\Core\Entity\EntityTypeManagerInterface) $entityTypeManager
   */
  public function __construct(
    private \Closure $entityTypeManager,
    private readonly PintoMapping $pintoMapping,
    #[Autowire(param: 'pinto_entity.mapping')]
    string $mapping,
  ) {
    $this->mappingByBundle = new \SplObjectStorage();
    // @phpstan-ignore-next-line
    $this->componentMapping = \unserialize($mapping);
  }

  public function view(EntityInterface $entity, string $view_mode = 'full', ?string $langcode = NULL): mixed {
    $entityStorage = ($this->entityTypeManager)()->getStorage($entity->getEntityTypeId());

    $thisBundle = $this->bundles[\sprintf('%s&%s', $entity->getEntityTypeId(), $entity->bundle())] ??= Bundle::fromEntity($entity);

    // Memoize for this request.
    $mapping = $this->mappingByBundle[$thisBundle] ??= (function () use ($entityStorage, $thisBundle) {
      $mappingByBundle = [];

      // Pinto component mapping.
      foreach ($this->componentMapping as [$bundle, $viewMode, $factoryMethod, $entityParams, $contextParams]) {
        if (FALSE === $thisBundle->equals($bundle)) {
          continue;
        }

        $mappingByBundle[$viewMode->viewMode] = [
          $viewMode,
          \ReflectionMethod::createFromMethodName($factoryMethod),
          $entityParams,
          $contextParams,
        ];
      }

      // Gather from bundle class, sadly this cannot be done in the container.
      $validFactoryReturnTypes = [];
      foreach ($this->pintoMapping->getResources() as $resource) {
        $componentClass = $resource->getClass();
        if (NULL !== $componentClass) {
          $validFactoryReturnTypes[] = $componentClass;
        }
      }

      /** @var class-string<\Drupal\Core\Entity\EntityInterface> $bundleClass */
      $bundleClass = $entityStorage->getEntityClass($thisBundle->bundle);
      $rClass = new \ReflectionClass($bundleClass);
      $fallback = new EntityView($thisBundle->entityType, [$thisBundle->bundle], 'full');
      foreach (EntityView::findOnClass($rClass, $validFactoryReturnTypes, $fallback) as [$bundle, $viewMode, $factoryMethod, $entityParams, $contextParams]) {
        if (FALSE === $thisBundle->equals($bundle)) {
          continue;
        }

        $mappingByBundle[$viewMode->viewMode] = [
          $viewMode,
          \ReflectionMethod::createFromMethodName($factoryMethod),
          $entityParams,
          $contextParams,
        ];
      }

      return $mappingByBundle;
    })();

    if (FALSE === \array_key_exists($view_mode, $mapping)) {
      return NULL;
    }

    [$viewMode, $rFactoryMethod, $entityParams, $contextParams] = $mapping[$view_mode];

    // Method args.
    $methodArguments = [];
    foreach ($entityParams as $paramName) {
      $methodArguments[$paramName] = $entity;
    }
    if ([] !== $contextParams) {
      $context = new EntityViewContext($entity, $viewMode);
      foreach ($contextParams as $paramName) {
        $methodArguments[$paramName] = $context;
      }
    }

    if ($rFactoryMethod->isStatic()) {
      // EntityView::findOnClass already ensures the return type is a known Pinto component.
      /** @var object $obj */
      $obj = $rFactoryMethod->invoke(NULL, ...$methodArguments);
    }
    else {
      throw new \LogicException('Unhandled!');
    }

    return $this->pintoMapping->getBuilder($obj)();
  }

}
