<?php

declare(strict_types=1);

namespace Drupal\pinto_entity;

use Drupal\canvas\EntityHandlers\ContentTemplateAwareViewBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityViewBuilderInterface;

/**
 * Modifies produced view controller using service decoration.
 *
 * Cleanly allows us to change the build by bundle, without changing the singular view builder of the entire entity type.
 *
 * Needs to `extend EntityTypeManager` temporarily until https://www.drupal.org/project/drupal/issues/3564264 is fixed.
 */
final class PintoEntityEntityTypeManager extends EntityTypeManager implements EntityTypeManagerInterface {

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line pluginManagerSetsCacheBackend.missingCacheBackend constructor.missingParentCall
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $inner,
    private readonly PintoEntity $pintoEntity,
  ) {
  }

  public function clearCachedDefinitions(): void {
    $this->inner->clearCachedDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function useCaches($use_caches = FALSE): void {
    $this->inner->useCaches($use_caches);
  }

  /**
   * {@inheritdoc}
   */
  public function hasDefinition($plugin_id) {
    return $this->inner->hasDefinition($plugin_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getAccessControlHandler($entity_type_id) {
    return $this->inner->getAccessControlHandler($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getStorage($entity_type_id) {
    return $this->inner->getStorage($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getViewBuilder($entity_type_id) {
    // @phpstan-ignore-next-line return.type
    return $this->getHandler($entity_type_id, 'view_builder');
  }

  /**
   * {@inheritdoc}
   */
  public function getListBuilder($entity_type_id) {
    return $this->inner->getListBuilder($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormObject($entity_type_id, $operation) {
    return $this->inner->getFormObject($entity_type_id, $operation);
  }

  /**
   * {@inheritdoc}
   */
  public function getRouteProviders($entity_type_id) {
    return $this->inner->getRouteProviders($entity_type_id);
  }

  /**
   * {@inheritdoc}
   */
  public function hasHandler($entity_type_id, $handler_type) {
    return $this->inner->hasHandler($entity_type_id, $handler_type);
  }

  /**
   * {@inheritdoc}
   */
  public function getHandler($entity_type_id, $handler_type) {
    $handler = $this->inner->getHandler($entity_type_id, $handler_type);

    // This Canvas $handler_type is a real terrible thing.
    if (($handler_type === 'view_builder') || (\class_exists(ContentTemplateAwareViewBuilder::class) && $handler_type === ContentTemplateAwareViewBuilder::DECORATED_HANDLER_KEY)) {
      \assert($handler instanceof EntityViewBuilderInterface);
      return new PintoEntityEntityViewBuilder(
        pintoEntity: $this->pintoEntity,
        original: $handler,
      );
    }

    return $handler;
  }

  /**
   * {@inheritdoc}
   */
  public function createHandlerInstance($class, ?EntityTypeInterface $definition = NULL) {
    return $this->inner->createHandlerInstance($class, $definition);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line method.childReturnType
   */
  public function getDefinition($entity_type_id, $exception_on_invalid = TRUE) {
    return $this->inner->getDefinition($entity_type_id, $exception_on_invalid);
  }

  /**
   * {@inheritdoc}
   *
   * @phpstan-ignore-next-line method.childReturnType
   */
  public function getDefinitions() {
    return $this->inner->getDefinitions();
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    return $this->inner->createInstance($plugin_id, $configuration);
  }

  /**
   * {@inheritdoc}
   */
  public function getInstance(array $options) {
    return $this->inner->getInstance($options);
  }

  /**
   * Default EntityTypeManager has some naughty methods not on interface.
   */
  public function getActiveDefinition($entity_type_id): EntityTypeInterface {
    // @phpstan-ignore-next-line method.notFound
    return $this->inner->getActiveDefinition($entity_type_id);
  }

}
