<?php

declare(strict_types=1);

namespace Drupal\pinto_entity\EntityView\Attribute;

use Drupal\Core\Entity\EntityInterface;
use Drupal\pinto_entity\EntityView\Bundle;
use Drupal\pinto_entity\EntityView\EntityViewContext;
use Drupal\pinto_entity\EntityView\ViewMode;
use Nette\Utils\Type;

/**
 * @phpstan-type EntityViewMapping array{Bundle, ViewMode, string, string[], string[]}
 */
#[\Attribute(flags: \Attribute::TARGET_METHOD)]
final class EntityView {

  private ?self $fallback = NULL;

  /**
   * Provide the entity type, and bundles and view modes.
   *
   * Entity type and bundle may be left empty when #[EntityView] is used on methods in a bundle class.
   *
   * @phpstan-param string|null $entityType
   * @phpstan-param string|string[]|null $bundles
   * @phpstan-param string|string[] $viewModes
   */
  public function __construct(
    private string|null $entityType = NULL,
    private string|array|null $bundles = NULL,
    private string|array|null $viewModes = NULL,
  ) {
    if (($entityType === NULL || $bundles === NULL || $viewModes === NULL) && $entityType !== $bundles) {
      throw new \LogicException('Both of $entityType and $bundles must be NULL, or not at all.');
    }
  }

  /**
   * @phpstan-return iterable<\Drupal\pinto_entity\EntityView\Bundle>
   */
  public function bundles(): iterable {
    return \array_map(
      callback: fn (string $bundle): Bundle => new Bundle(entityType: $this->entityType ?? ($this->fallback->entityType ?? throw new \LogicException('No fallback entity type available when not used with a bundle class')), bundle: $bundle),
      array: (array) ($this->bundles ?? ($this->fallback->bundles ?? throw new \LogicException('No fallback bundle available when not used with a bundle class'))),
    );
  }

  /**
   * @phpstan-return iterable<\Drupal\pinto_entity\EntityView\ViewMode>
   */
  public function viewModes(): iterable {
    return \array_map(
      callback: static fn (string $viewMode): ViewMode => new ViewMode(viewMode: $viewMode),
      array: (array) ($this->viewModes ?? ($this->fallback->viewModes ?? throw new \LogicException('No fallback view mode available when not used with a bundle class'))),
    );
  }

  /**
   * @phpstan-return iterable<Bundle, ViewMode>
   */
  public function all(): iterable {
    foreach ($this->bundles() as $bundle) {
      foreach ($this->viewModes() as $viewMode) {
        yield $bundle => $viewMode;
      }
    }
  }

  /**
   * @phpstan-param \ReflectionClass<covariant object> $rClass
   * @phpstan-param array<class-string<object>> $validFactoryReturnTypes
   * @phpstan-return iterable<EntityViewMapping>
   */
  public static function findOnClass(\ReflectionClass $rClass, array $validFactoryReturnTypes, ?self $fallback = NULL): iterable {
    // All must be public static.
    foreach ($rClass->getMethods(\ReflectionMethod::IS_PUBLIC) as $rMethod) {
      $attributes = $rMethod->getAttributes(EntityView::class);

      if ([] === $attributes) {
        continue;
      }

      if (FALSE === $rMethod->isStatic()) {
        throw new \LogicException(\sprintf('Method %s::%s() using %s attribute is expected to be public and static.', $rMethod->getDeclaringClass()->getName(), $rMethod->getName(), EntityView::class));
      }

      $validReturnType = FALSE;
      foreach ($validFactoryReturnTypes as $factoryReturnType) {
        if (Type::fromString($factoryReturnType)->allows(Type::fromReflection($rMethod) ?? throw new \LogicException('Unable to determine reflection for method'))) {
          $validReturnType = TRUE;
          break;
        }
      }

      if (FALSE === $validReturnType) {
        throw new \LogicException(\sprintf('The return type for %s::%s() is not typed with a known Pinto component. Allowed types may extend any one of %s', $rMethod->getDeclaringClass()->getName(), $rMethod->getName(), \implode(', ', $validFactoryReturnTypes)));
      }

      $factoryMethod = \sprintf('%s::%s', $rMethod->getDeclaringClass()->getName(), $rMethod->getName());
      $entityParams = static::findEntityParameters($rMethod);
      $contextParams = static::findEntityViewContextParameters($rMethod);

      foreach ($attributes as $attribute) {
        /** @var EntityView $entityView */
        $entityView = $attribute->newInstance();
        $entityView->fallback = $fallback;
        foreach ($entityView->all() as $bundle => $viewMode) {
          yield [$bundle, $viewMode, $factoryMethod, $entityParams, $contextParams];
        }
      }
    }
  }

  /**
   * @phpstan-return string[]
   */
  private static function findEntityParameters(\ReflectionMethod $rMethod): array {
    $parameters = [];

    foreach ($rMethod->getParameters() as $rParam) {
      // Nette is used to resolve types like `self`.
      if (Type::fromString(EntityInterface::class)->allows(Type::fromReflection($rParam) ?? throw new \LogicException('Unable to determine reflection for parameter'))) {
        $parameters[] = $rParam->getName();
      }
    }

    return $parameters;
  }

  /**
   * @phpstan-return string[]
   */
  private static function findEntityViewContextParameters(\ReflectionMethod $rMethod): array {
    $parameters = [];

    foreach ($rMethod->getParameters() as $rParam) {
      $paramType = $rParam->getType();
      if ($paramType instanceof \ReflectionNamedType && $paramType->getName() === EntityViewContext::class) {
        $parameters[] = $rParam->getName();
      }
    }

    return $parameters;
  }

}
