<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\DrupalLayoutDiscovery;

use Drupal\Core\Layout\LayoutDefault;
use Drupal\Core\Layout\LayoutDefinition;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\pinto_layout\Discovery\FrozenLayoutDefinition;
use Drupal\pinto_layout\Hook\PintoLayoutHooks;
use Pinto\List\ObjectListInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Pinto Layout plugin.
 *
 * This is not a part of regular plugin discovery as we do our own thing.
 *
 * @todo stop extending LayoutDefault
 *
 * @internal
 */
final class PintoLayout extends LayoutDefault implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    LayoutDefinition $plugin_definition,
    private DrupalLayoutDefinitionRepository $drupalLayoutDefinitionRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\Core\Layout\LayoutDefinition $plugin_definition */
    return new static($configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(DrupalLayoutDefinitionRepository::class),
    );
  }

  public function build(array $regions): array {
    /** @var array<string, array<mixed>> $regions */
    $layoutDefinition = $this->getFrozenDefinition();

    $build = [
      '#theme' => PintoLayoutHooks::FAUX_THEME_ELEMENT,
      '#' . PintoLayoutHooks::FROZEN_DEFINITION_RENDER_KEY => $layoutDefinition,
    ];

    // LB expects a certain structure:
    // We cannot clear out region data (blocks) here as LayoutBuilder::buildAdministrativeSection needs to decorate each block with attributes. (`build[$region][$uuid]['#attributes']['class'][]` etc).
    foreach ($layoutDefinition->regions->regions as $regionName) {
      if (\array_key_exists($regionName, $regions)) {
        $build[$regionName] = $regions[$regionName];
      }
    }

    return $build;
  }

  private function getFrozenDefinition(): FrozenLayoutDefinition {
    $pintoEnum = $this->getPluginDefinition()->get(DrupalLayoutDefinitionRepository::LAYOUT_PLUGIN_ADDITIONAL_PINTO_ENUM);
    if (!$pintoEnum instanceof ObjectListInterface) {
      // As set by  DrupalLayoutDefinitionRepository::getDefinitionsAsHookLayouts.
      throw new \LogicException('Impossible! Layout definition missing Pinto object enum');
    }

    return $this->drupalLayoutDefinitionRepository->getFrozenDefinitionByEnum($pintoEnum);
  }

}
