<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\PintoLayout\External;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\pinto_layout\Attribute\Regions;
use Drupal\pinto_layout\Discovery\FrozenLayoutDefinition;
use Drupal\pinto_layout\Discovery\RegionDataMode;
use Pinto\List\ObjectListInterface;

final class ExternallyDefined {

  private function __construct(
    private string $id,
    private TranslatableMarkup $label,
    private readonly ObjectListInterface $pintoEnum,
    private readonly Regions $regions,
    private \ReflectionMethod $factoryMethod,
  ) {
  }

  /**
   * @phpstan-param Regions|list<string|\Drupal\pinto_layout\Attribute\Region> $regions
   */
  public static function create(
    string $id,
    TranslatableMarkup $label,
    ObjectListInterface $pintoEnum,
    Regions|array $regions,
    callable $factoryMethod,
  ): static {
    /** @var array|string|\Closure $factoryMethod */
    // @phpstan-ignore-next-line varTag.nativeType
    if ($factoryMethod instanceof \Closure) {
      throw new \Exception('$factoryMethod must be statically callable.');
    }

    $rMethod = \is_array($factoryMethod)
      ? new \ReflectionMethod($factoryMethod[0], $factoryMethod[1])
      : \ReflectionMethod::createFromMethodName($factoryMethod);

    return new static(
      id: $id,
      label: $label,
      pintoEnum: $pintoEnum,
      regions: \is_array($regions) ? new Regions(regions: $regions) : $regions,
      factoryMethod: $rMethod,
    );
  }

  /**
   * @internal
   */
  public function freeze(): FrozenLayoutDefinition {
    return new FrozenLayoutDefinition(
      id: $this->id,
      label: $this->label,
      pintoEnum: $this->pintoEnum,
      regions: $this->regions,
      factoryMethod: $this->factoryMethod,
      regionDataMode: RegionDataMode::AsLayoutDataSingleParameter,
      regionAttributesAsParameter: [],
    );
  }

}
