<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\Discovery;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\pinto_layout\Attribute\LayoutDefinition;
use Drupal\pinto_layout\Attribute\Region;
use Drupal\pinto_layout\Attribute\Regions;
use Drupal\pinto_layout\PintoLayout\Data\LayoutData;
use Drupal\pinto_layout\PintoLayout\Data\RegionAttributes;
use Drupal\pinto_layout\PintoLayout\PintoLayoutInterface;
use Pinto\Attribute\Definition;
use Pinto\PintoMapping;
use Pinto\Slots\Origin\Parameter;

/**
 * @internal
 */
final class PintoLayoutDiscovery {

  public function __construct(
    private readonly PintoMapping $pintoMapping,
  ) {
  }

  /**
   * @phpstan-return \Generator<\Drupal\pinto_layout\Discovery\FrozenLayoutDefinition>
   * @internal
   */
  public function layoutDefinitionsFromPintoObjects(): \Generator {
    foreach ($this->pintoMapping->getEnumClasses() as $pintoList) {
      $rPintoList = new \ReflectionClass($pintoList);
      foreach ($rPintoList->getReflectionConstants() as $constant) {
        /** @var array<\ReflectionAttribute<\Pinto\Attribute\Definition>> $attributes */
        $attributes = $constant->getAttributes(Definition::class);
        $definition = ($attributes[0] ?? NULL)?->newInstance();
        if ($definition === NULL) {
          continue;
        }

        $enum = $this->pintoMapping->getByClass($definition->className);

        $rObjectClass = new \ReflectionClass($definition->className);

        /** @var \Drupal\pinto_layout\Attribute\LayoutDefinition|null $layoutDefinition */
        $layoutDefinition = ($rObjectClass->getAttributes(LayoutDefinition::class)[0] ?? NULL)?->newInstance();

        $layoutId = $layoutDefinition->id ?? \strtolower('pinto_layout__' . $rObjectClass->getShortName());
        $layoutLabel = $layoutDefinition->label ?? new TranslatableMarkup('Layout from @className', ['@className' => $rObjectClass->getName()]);

        if ($layoutDefinition?->regions !== NULL && $layoutDefinition->regions->regions !== []) {
          yield new FrozenLayoutDefinition(
            $layoutId,
            $layoutLabel,
            $enum,
            $layoutDefinition->regions,
            ...static::determineFactoryMethod($rObjectClass),
          );
          continue;
        }

        $slotParameterRegions = $this->trySlotRegions($definition->className);
        // If there were slot regions, use them.
        if ($slotParameterRegions !== NULL) {
          yield new FrozenLayoutDefinition(
            $layoutId,
            $layoutLabel,
            $enum,
            $slotParameterRegions[1],
            ...static::determineFactoryMethod($rObjectClass, $slotParameterRegions[0]),
          );
          continue;
        }

        /** @var \Drupal\pinto_layout\Attribute\Regions|null $regions */
        $regions = ($rObjectClass->getAttributes(Regions::class)[0] ?? NULL)?->newInstance();
        if ($regions !== NULL) {
          yield new FrozenLayoutDefinition(
            $layoutId,
            $layoutLabel,
            $enum,
            $regions,
            ...static::determineFactoryMethod($rObjectClass),
          );
        }
      }
    }
  }

  /**
   * @phpstan-param \ReflectionClass<object> $rObjectClass
   * @phpstan-return array{
   *   factoryMethod: \ReflectionFunctionAbstract,
   *   regionDataMode: RegionDataMode,
   *   regionAttributesAsParameter: string[],
   *   frozenLayoutDefinitionAsParameter: string[],
   *   layoutDataAsParameter: string[],
   * }
   */
  public static function determineFactoryMethod(\ReflectionClass $rObjectClass, \ReflectionFunctionAbstract ...$tryMethods): array {
    /** @var \Drupal\pinto_layout\Attribute\LayoutDefinition|null $layoutDefinition */
    $layoutDefinition = ($rObjectClass->getAttributes(LayoutDefinition::class)[0] ?? NULL)?->newInstance();
    if ($layoutDefinition?->factoryMethod !== NULL) {
      if (!$rObjectClass->hasMethod($layoutDefinition->factoryMethod)) {
        throw new \Exception(\sprintf('%s::%s() referenced by %s on %s does not exist', $rObjectClass->getName(), $layoutDefinition->factoryMethod, LayoutDefinition::class, $rObjectClass->getName()));
      }

      $method = $rObjectClass->getMethod($layoutDefinition->factoryMethod);
      if ($method->isStatic() === FALSE || $method->isAbstract() || $method->isProtected()) {
        throw new \Exception(\sprintf('%s::%s() referenced by %s on %s must be a public static method.', $rObjectClass->getName(), $layoutDefinition->factoryMethod, LayoutDefinition::class, $rObjectClass->getName()));
      }

      $hasLayoutDataParams = static::findLayoutDataParameters($method) !== [];

      return [
        'factoryMethod' => $method,
        'regionDataMode' => $hasLayoutDataParams ? RegionDataMode::AsLayoutDataSingleParameter : RegionDataMode::AsParameterNamesFromRegionNames,
        'regionAttributesAsParameter' => $hasLayoutDataParams ? [] : static::findRegionAttributesParameters($method),
        'frozenLayoutDefinitionAsParameter' => $hasLayoutDataParams ? [] : static::findLayoutDefinitionParameters($method),
        'layoutDataAsParameter' => static::findLayoutDataParameters($method),
      ];
    }

    if ($rObjectClass->implementsInterface(PintoLayoutInterface::class)) {
      return [
        'factoryMethod' => $rObjectClass->getMethod('createForLayout'),
        'regionDataMode' => RegionDataMode::AsLayoutDataSingleParameter,
        'regionAttributesAsParameter' => [],
        'frozenLayoutDefinitionAsParameter' => [],
        'layoutDataAsParameter' => [],
      ];
    }

    foreach ($tryMethods as $tryMethod) {
      if (($tryMethod instanceof \ReflectionMethod && $tryMethod->isPublic()) && ($tryMethod->getName() === '__construct' || $tryMethod->isStatic())) {
        return [
          'factoryMethod' => $tryMethod,
          'regionDataMode' => RegionDataMode::AsParameterNamesFromRegionNames,
          'regionAttributesAsParameter' => static::findRegionAttributesParameters($tryMethod),
          'frozenLayoutDefinitionAsParameter' => static::findLayoutDefinitionParameters($tryMethod),
          'layoutDataAsParameter' => static::findLayoutDataParameters($tryMethod),
        ];
      }
    }

    throw new \Exception('Unable to determine factory method for ' . $rObjectClass->getName() . '. If you used `#[Region]` on slot parameters, make sure all non-`#[Region]` parameters have a default value. If thats not possible, use `#[Regions]` on the class or a method instead.');
  }

  /**
   * @phpstan-return string[]
   */
  private static function findRegionAttributesParameters(\ReflectionMethod $rMethod): array {
    $parameters = [];

    foreach ($rMethod->getParameters() as $rParam) {
      $paramType = $rParam->getType();
      if ($paramType instanceof \ReflectionNamedType && $paramType->getName() === RegionAttributes::class) {
        $parameters[] = $rParam->getName();
      }
    }

    return $parameters;
  }

  /**
   * @phpstan-return string[]
   */
  private static function findLayoutDefinitionParameters(\ReflectionMethod $rMethod): array {
    $parameters = [];

    foreach ($rMethod->getParameters() as $rParam) {
      $paramType = $rParam->getType();
      if ($paramType instanceof \ReflectionNamedType && $paramType->getName() === FrozenLayoutDefinition::class) {
        $parameters[] = $rParam->getName();
      }
    }

    return $parameters;
  }

  /**
   * @phpstan-return string[]
   */
  private static function findLayoutDataParameters(\ReflectionMethod $rMethod): array {
    $parameters = [];

    foreach ($rMethod->getParameters() as $rParam) {
      $paramType = $rParam->getType();
      if ($paramType instanceof \ReflectionNamedType && $paramType->getName() === LayoutData::class) {
        $parameters[] = $rParam->getName();
      }
    }

    return $parameters;
  }

  /**
   * Determine if a theme object uses slots and has #[Region] attributes.
   *
   * Determines if the theme objects is #[Slots] based, and if so, checks if any of the method-parameter derived slots
   * have a #[Region] attribute. If there is at least one, that method is marked as a possible factory.
   *
   * @phpstan-param class-string $objectClassName
   * @phpstan-return array{\ReflectionFunctionAbstract, Regions}|null
   */
  private function trySlotRegions(string $objectClassName): ?array {
    $slotParameterRegions = [];

    $def = $this->pintoMapping->getThemeDefinition($objectClassName);
    if (!$def instanceof \Pinto\Slots\Definition) {
      return NULL;
    }

    /** @var list<\ReflectionFunctionAbstract> $parameterFn */
    $parameterFn = [];
    foreach ($def->slots as $slot) {
      $o = $slot->origin;
      if ($o instanceof Parameter) {
        $pr = $o->parameterReflection();
        $parameterFn[] = $pr->getDeclaringFunction();
        $region = ($pr->getAttributes(Region::class)[0] ?? NULL)?->newInstance();
        if ($region !== NULL) {
          $region->name ??= $pr->getName();
          $slotParameterRegions[$pr->getName()] = $region;
        }
      }
    }

    // Sanity check only zero or one method defines the slots.
    if (\count(\array_unique($parameterFn)) > 1) {
      throw new \LogicException('Somehow the slots were defined by different functions (methods)');
    }

    if ($parameterFn === [] || $slotParameterRegions === []) {
      return NULL;
    }

    $factoryMethod = $parameterFn[0];

    // Validate if there is at least one #[Region] on a slot, then all *required* parameters must also be #[Region].
    $requiredParametersNotRegions = [];
    foreach ($factoryMethod->getParameters() as $rParam) {
      if (!\array_key_exists($rParam->getName(), $slotParameterRegions)) {
        $requiredParametersNotRegions[] = $rParam;
      }
    }

    foreach ($requiredParametersNotRegions as $k => $rParam) {
      // Optional parameters do not need to be regions if they can supply their own default values.
      if ($rParam->isOptional()) {
        unset($requiredParametersNotRegions[$k]);
      }

      // Also ignore anything typed with #[RegionAttributes] as we can populate these automatically.
      $paramType = $rParam->getType();
      if ($paramType instanceof \ReflectionNamedType && $paramType->getName() === RegionAttributes::class) {
        unset($requiredParametersNotRegions[$k]);
      }
    }

    // If there is anything less, then we cannot built this object programmatically. So quit.
    if ($requiredParametersNotRegions !== []) {
      return NULL;
    }

    return [
      $factoryMethod,
      new Regions(\array_values($slotParameterRegions)),
    ];
  }

}
