<?php

declare(strict_types=1);

namespace Drupal\Tests\pinto_layout\Functional;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionComponent;
use Drupal\layout_builder\SectionListInterface;
use Drupal\pinto_layout_support\Plugin\Block\PintoLayoutTestBlock;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\pinto_layout\PintoLayoutFixtures;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * @group pinto_layout
 */
final class PintoLayoutTest extends BrowserTestBase {

  protected $defaultTheme = 'stark';

  protected static $modules = [
    'entity_test',
    'pinto_layout_support',
    'pinto_layout_test',
    'layout_discovery',
    'layout_builder',
    'pinto_layout',
    'pinto',
  ];

  protected function setUp(): void {
    parent::setUp();

    $display = LayoutBuilderEntityViewDisplay::create([
      'targetEntityType' => 'entity_test',
      'bundle' => 'entity_test',
      'mode' => 'default',
      'status' => TRUE,
    ]);

    $display
      ->enableLayoutBuilder()
      ->setOverridable()
      ->save();

    (Role::load(RoleInterface::ANONYMOUS_ID) ?? throw new \LogicException('Impossible'))
      ->grantPermission('view test entity')
      ->save();
  }

  /**
   * Render out all test layouts, including region content, into a Layout Builder entity.
   */
  public function testLayouts(): void {
    $e = EntityTest::create();

    $expectedContent = [];
    foreach (PintoLayoutFixtures::ExpectedLayouts as $layoutId => $regions) {
      $layout = $e->get(OverridesSectionStorage::FIELD_NAME);
      static::assertInstanceOf(SectionListInterface::class, $layout);
      $layout->appendSection($section = new Section($layoutId, layout_settings: []));

      self::assertGreaterThan(0, \count($regions));
      foreach ($regions as $region) {
        $expectedContent[] = $content = \sprintf('%s--%s--Content!', $layoutId, $region);
        $section->appendComponent(new SectionComponent(static::uuidService()->generate(), region: $region, configuration: [
          'id' => PintoLayoutTestBlock::PLUGIN_ID,
          'label_display' => FALSE,
          PintoLayoutTestBlock::CONFIG_CUSTOM_CONTENT_KEY => $content,
        ]));
      }
    }

    $e->save();

    $this->drupalGet($e->toUrl());

    // Assert the slot contents is visible. We don't necessarily care about HTML/element structure at this point.
    foreach ($expectedContent as $content) {
      $this->assertSession()->responseContains($content);
    }
  }

  private static function uuidService(): UuidInterface {
    /** @var \Drupal\Component\Uuid\UuidInterface */
    return \Drupal::service(UuidInterface::class);
  }

}
