<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\DrupalLayoutDiscovery;

use Drupal\Core\Layout\LayoutDefinition;
use Drupal\pinto_layout\Discovery\FrozenLayoutDefinition;
use Drupal\pinto_layout\Discovery\FrozenRegion;
use Drupal\pinto_layout\Discovery\PintoLayoutDiscovery;
use Drupal\pinto_layout\PintoLayout\External\ExternallyDefined;
use Pinto\PintoMapping;
use Symfony\Component\DependencyInjection\ServiceLocator;

/**
 * Discovers and compiles FrozenLayoutDefinition into Drupal hooks.
 *
 * @internal
 */
final class DrupalLayoutDefinitionRepository {

  public const LAYOUT_PLUGIN_ADDITIONAL_PINTO_ENUM = 'LAYOUT_PLUGIN_ADDITIONAL_PINTO_ENUM';
  public const LAYOUT_PLUGIN_ADDITIONAL_FORMS = 'LAYOUT_PLUGIN_ADDITIONAL_FORMS';

  /**
   * @phpstan-param \Symfony\Component\DependencyInjection\ServiceLocator<\Drupal\pinto_layout\PintoLayout\External\ExternallyDefinedInterface> $externallyDefinedLocator
   */
  public function __construct(
    private readonly PintoMapping $pintoMapping,
    private readonly ServiceLocator $externallyDefinedLocator,
  ) {
  }

  /**
   * @phpstan-return iterable<\Drupal\pinto_layout\Discovery\FrozenLayoutDefinition>
   */
  public function getDefinitions(): iterable {
    yield from (new PintoLayoutDiscovery($this->pintoMapping))
      ->layoutDefinitionsFromPintoObjects();

    foreach (\array_keys($this->externallyDefinedLocator->getProvidedServices()) as $id) {
      $service = $this->externallyDefinedLocator->get($id);
      yield from \array_map(
        static fn (ExternallyDefined $externallyDefined): FrozenLayoutDefinition => $externallyDefined->freeze(),
        \iterator_to_array($service->getDefinitions()),
      );
    }
  }

  /**
   * @phpstan-return array<string, \Drupal\Core\Layout\LayoutDefinition>
   */
  public function getDefinitionsAsHookLayouts(): array {
    $definitions = [];

    foreach ($this->getDefinitions() as $frozenLayoutDefinition) {
      // Core layouts is #[Layout]->get() into get_object_vars().
      $definitions[$frozenLayoutDefinition->layoutId()] = new LayoutDefinition([
        // Any key here can match any of the parameters from \Drupal\Core\Layout\Attribute\Layout::__construct().
        'id' => $frozenLayoutDefinition->layoutId(),
        'label' => $frozenLayoutDefinition->layoutLabel(),
        'class' => PintoLayout::class,
        'regions' => \array_reduce(
          \iterator_to_array($frozenLayoutDefinition->regions->regionDefinitions()),
          static function (array $regions, FrozenRegion $region) {
            $regions[$region->name] = ['label' => $region->label];
            return $regions;
          },
          [],
        ),
        'additional' => [
          static::LAYOUT_PLUGIN_ADDITIONAL_PINTO_ENUM => $frozenLayoutDefinition->pintoEnum,
          static::LAYOUT_PLUGIN_ADDITIONAL_FORMS => $frozenLayoutDefinition->forms,
        ],
      ]);
    }

    return $definitions;
  }

  /**
   * @throws \InvalidArgumentException
   *   When an enum is not associated with a Pinto Layout.
   */
  public function getFrozenDefinitionByEnum(\Pinto\List\ObjectListInterface $pintoObjectEnum): FrozenLayoutDefinition {
    foreach ($this->getDefinitions() as $frozenLayoutDefinition) {
      if ($frozenLayoutDefinition->pintoEnum === $pintoObjectEnum) {
        return $frozenLayoutDefinition;
      }
    }

    throw new \InvalidArgumentException($pintoObjectEnum::class . '::' . $pintoObjectEnum->name . ' is not associated with a layout definition.');
  }

}
