<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\PintoLayout\Data;

use Drupal\Core\Template\Attribute;
use Drupal\pinto_layout\Discovery;
use Drupal\pinto_layout\Hook\PintoLayoutHooks;

/**
 * Region attributes from layout builder editor.
 */
final class RegionAttributes {

  /**
   * @phpstan-param array<string, Attribute> $regionAttributes
   */
  private function __construct(
    private Attribute $containerAttributes,
    private array $regionAttributes = [],
  ) {
  }

  /**
   * Just like \template_preprocess_layout() juggle things into a `region_attributes` Twig var.
   *
   * Instead, we pass this along as a well formed object so the Pinto object can decide how to handle the data and name its region attribute slots.
   *
   * @phpstan-param array{'pinto_layout_render_element': array<string, array<string, mixed>>} $variables
   * @see \template_preprocess_layout()
   *
   * @internal
   */
  public static function fromVariables(Discovery\FrozenLayoutDefinition $definition, $variables): static {
    $regionAttributes = [];
    foreach ($definition->regions->regions as $regionName) {
      // @phpstan-ignore-next-line argument.type
      $regionAttributes[$regionName] = new Attribute($variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT][$regionName]['#attributes'] ?? []);
    }
    return new static(
      containerAttributes: new Attribute($variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT]['#attributes'] ?? []),
      regionAttributes: $regionAttributes,
    );
  }

  /**
   * As an array suitable and compatible with Layout Discovery's region_attributes for use as a Twig variable.
   *
   * @see \template_preprocess_layout()
   *
   * @return array<string, \Drupal\Core\Template\Attribute>
   *   Attribute objects keyed by slot name.
   */
  public function regionsAsArray(): array {
    return $this->regionAttributes;
  }

  public function containerAttributes(): Attribute {
    return $this->containerAttributes;
  }

  /**
   * @throws \InvalidArgumentException
   *   When the region does not exist.
   */
  public function regionAttributes(string $regionName): Attribute {
    return $this->regionAttributes[$regionName] ?? throw new \InvalidArgumentException('Region `' . $regionName . '` does not exist.');
  }

}
