<?php

declare(strict_types=1);

namespace Drupal\pinto_layout_test\Forms;

use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\PluginFormBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * @property \Drupal\pinto_layout\DrupalLayoutDiscovery\PintoLayout<Configuration> $plugin
 * @phpstan-type Configuration array{color: ?string, size: ?string}
 */
final class LayoutFormsForm extends PluginFormBase implements ContainerInjectionInterface {

  use AutowireTrait;
  use StringTranslationTrait;

  private SubformForm $subform;

  public function __construct(
    TranslationInterface $translation,
  ) {
    $this->setStringTranslation($translation);
    $this->subform = new SubformForm();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = $this->plugin->buildConfigurationForm($form, $form_state);

    $config = $this->plugin->getConfiguration();

    $form['color'] = [
      '#title' => $this->t('Color'),
      '#description' => $this->t('A CSS color, e.g <code>#F00</code>, or <code>rebeccapurple</code>.'),
      '#type' => 'textfield',
      '#default_value' => $config['color'] ?? '',
    ];

    $form['sub'] = [];
    $this->subform->size = $config['size'] ?? NULL;
    $subformState = SubformState::createForSubform($form['sub'], $form, $form_state);
    /** @var array{sub: array} $form */
    $form['sub'] = $this->subform->buildConfigurationForm($form['sub'], $subformState);

    return $form;
  }

  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    /** @var array{sub: array} $form */
    $this->plugin->validateConfigurationForm($form, $form_state);

    $subformState = SubformState::createForSubform($form['sub'], $form, $form_state);

    /** @var array{sub: array} $form */
    $this->subform->validateConfigurationForm($form['sub'], $subformState);
  }

  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    /** @var array{sub: array} $form */

    $subformState = SubformState::createForSubform($form['sub'], $form, $form_state);
    $this->subform->submitConfigurationForm($form['sub'], $subformState);

    /** @var string|null $color */
    $color = $form_state->getValue('color');
    $this->plugin->setConfiguration([
      'color' => $color,
      'size' => $this->subform->size,
    ]);

    $this->plugin->submitConfigurationForm($form, $form_state);
  }

}
