<?php

declare(strict_types=1);

namespace Drupal\Tests\pinto_layout\Functional;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Url;
use Drupal\entity_test\Entity\EntityTest;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay;
use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\layout_builder\Section;
use Drupal\layout_builder\SectionComponent;
use Drupal\layout_builder\SectionListInterface;
use Drupal\pinto_layout_support\Plugin\Block\PintoLayoutTestBlock;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\pinto_layout\PintoLayoutFixtures;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * @group pinto_layout
 */
final class PintoLayoutTest extends BrowserTestBase {

  protected $defaultTheme = 'stark';

  protected static $modules = [
    'entity_test',
    'pinto_layout_support',
    'pinto_layout_test',
    'layout_discovery',
    'layout_builder',
    'pinto_layout',
    'pinto',
  ];

  protected function setUp(): void {
    parent::setUp();

    $display = LayoutBuilderEntityViewDisplay::create([
      'targetEntityType' => 'entity_test',
      'bundle' => 'entity_test',
      'mode' => 'default',
      'status' => TRUE,
    ]);

    $display
      ->enableLayoutBuilder()
      ->setOverridable()
      ->save();

    (Role::load(RoleInterface::ANONYMOUS_ID) ?? throw new \LogicException('Impossible'))
      ->grantPermission('view test entity')
      ->grantPermission('configure all entity_test entity_test layout overrides')
      ->save();
  }

  /**
   * Render out all test layouts, including region content, into a Layout Builder entity.
   */
  public function testLayouts(): void {
    $e = EntityTest::create();

    $expectedContent = [];
    foreach (PintoLayoutFixtures::ExpectedLayouts as $layoutId => $regions) {
      $layout = $e->get(OverridesSectionStorage::FIELD_NAME);
      static::assertInstanceOf(SectionListInterface::class, $layout);
      $layout->appendSection($section = new Section($layoutId, layout_settings: []));

      self::assertGreaterThan(0, \count($regions));
      foreach ($regions as $region) {
        $expectedContent[] = $content = \sprintf('%s--%s--Content!', $layoutId, $region);
        $section->appendComponent(new SectionComponent(static::uuidService()->generate(), region: $region, configuration: [
          'id' => PintoLayoutTestBlock::PLUGIN_ID,
          'label_display' => FALSE,
          PintoLayoutTestBlock::CONFIG_CUSTOM_CONTENT_KEY => $content,
        ]));
      }
    }

    $e->save();

    $this->drupalGet($e->toUrl());

    // Assert the slot contents is visible. We don't necessarily care about HTML/element structure at this point.
    foreach ($expectedContent as $content) {
      $this->assertSession()->responseContains($content);
    }
  }

  /**
   * Test layout with forms.
   */
  public function testLayoutPluginForm(): void {
    $layoutId = 'pinto_layout__layoutforms';

    $e = EntityTest::create(['name' => 'LayoutBuilderController::title assert() needs a label']);
    $layout = $e->get(OverridesSectionStorage::FIELD_NAME);
    static::assertInstanceOf(SectionListInterface::class, $layout);
    $layout->appendSection($section = new Section($layoutId, layout_settings: [
      'label' => 'My Section',
    ]));
    $e->save();

    $url = Url::fromRoute(\sprintf('layout_builder.overrides.%s.view', $e->getEntityTypeId()), [
      $e->getEntityTypeId() => $e->id(),
    ]);
    $this->drupalGet($url);
    $this->assertSession()->linkExists('Configure My Section');

    $formUrl = Url::fromRoute('layout_builder.configure_section', route_parameters: [
      'section_storage_type' => 'overrides',
      'section_storage' => \sprintf('%s.%s', $e->getEntityTypeId(), $e->id()),
      'delta' => '0',
      // Not required, see route definition.
      'plugin_id' => NULL,
    ]);
    $this->drupalGet($formUrl);

    $this->assertSession()->fieldValueEquals('layout_settings[label]', 'My Section');
    // No values yet.
    $this->assertSession()->fieldExists('layout_settings[color]');
    $this->assertSession()->fieldExists('layout_settings[sub][size]');
    $this->assertSession()->buttonExists('Update');

    $this->submitForm(edit: [
      'layout_settings[label]' => 'My New Section Label',
      'layout_settings[color]' => 'purple',
      'layout_settings[sub][size]' => '16',
    ], submit: 'Update');
    // Confirm redirected:
    $this->assertSession()->pageTextContains('You are editing the layout for this Entity Test Bundle test entity.');
    // Section link text was updated with new label:
    $this->assertSession()->linkExists('Configure My New Section Label');
    $this->getSession()->getPage()->pressButton('Save layout');

    // Reload.
    /** @var \Drupal\entity_test\Entity\EntityTest $e */
    $e = $e::load($e->id());
    $layout = $e->get(OverridesSectionStorage::FIELD_NAME);
    static::assertInstanceOf(SectionListInterface::class, $layout);
    $section = $layout->getSection(0);
    static::assertEquals('pinto_layout__layoutforms', $section->getLayoutId());
    static::assertEquals([
      'label' => 'My New Section Label',
      'color' => 'purple',
      'size' => '16',
      'context_mapping' => [],
    ], $section->getLayoutSettings());

    // Ensure the settings bubbled all the way through the object into the Twig template:
    $this->assertSession()->elementsCount('css', '.pinto-layout-container', 1);
    $this->assertSession()->elementAttributeContains('css', '.pinto-layout-container', 'style', 'background-color: purple; border: 16px solid black;');
  }

  /**
   * Test loading a form of a layout without forms doesn't throw exceptions.
   */
  public function testLayoutPluginFormNoForms(): void {
    $layoutId = 'pinto_layout__layoutlabel';

    $e = EntityTest::create(['name' => 'LayoutBuilderController::title assert() needs a label']);
    $layout = $e->get(OverridesSectionStorage::FIELD_NAME);
    static::assertInstanceOf(SectionListInterface::class, $layout);
    $layout->appendSection($section = new Section($layoutId, layout_settings: [
      'label' => 'My Section',
    ]));
    $e->save();

    $url = Url::fromRoute(\sprintf('layout_builder.overrides.%s.view', $e->getEntityTypeId()), [
      $e->getEntityTypeId() => $e->id(),
    ]);
    $this->drupalGet($url);
    $this->assertSession()->linkExists('Configure My Section');

    $formUrl = Url::fromRoute('layout_builder.configure_section', route_parameters: [
      'section_storage_type' => 'overrides',
      'section_storage' => \sprintf('%s.%s', $e->getEntityTypeId(), $e->id()),
      'delta' => '0',
      // Not required, see route definition.
      'plugin_id' => NULL,
    ]);
    $this->drupalGet($formUrl);

    $this->assertSession()->fieldValueEquals('layout_settings[label]', 'My Section');
  }

  private static function uuidService(): UuidInterface {
    /** @var \Drupal\Component\Uuid\UuidInterface */
    return \Drupal::service(UuidInterface::class);
  }

}
