<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\DrupalLayoutDiscovery;

use Drupal\Core\Layout\LayoutDefault;
use Drupal\Core\Layout\LayoutDefinition;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginWithFormsInterface;
use Drupal\pinto_layout\Discovery\FrozenLayoutDefinition;
use Drupal\pinto_layout\Hook\PintoLayoutHooks;
use Drupal\pinto_layout\PintoLayout\Data\PluginConfiguration;
use Pinto\List\ObjectListInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Pinto Layout plugin.
 *
 * This is not a part of regular plugin discovery as we do our own thing.
 *
 * @todo stop extending LayoutDefault
 *
 * @internal
 *
 * @template Configuration of array<mixed>
 * @method Configuration getConfiguration()
 * @method void setConfiguration(Configuration $configuration)
 */
final class PintoLayout extends LayoutDefault implements ContainerFactoryPluginInterface, PluginWithFormsInterface {

  public function __construct(
    array $configuration,
    string $plugin_id,
    LayoutDefinition $plugin_definition,
    private DrupalLayoutDefinitionRepository $drupalLayoutDefinitionRepository,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Ignore regression from https://www.drupal.org/project/drupal/issues/3452852.
   *
   * @phpstan-ignore-next-line method.childParameterType
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\Core\Layout\LayoutDefinition $plugin_definition */
    return new static($configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(DrupalLayoutDefinitionRepository::class),
    );
  }

  public function build(array $regions): array {
    /** @var array<string, array<mixed>> $regions */
    $layoutDefinition = $this->getFrozenDefinition();

    $build = [
      '#theme' => PintoLayoutHooks::FAUX_THEME_ELEMENT,
      '#' . PintoLayoutHooks::FROZEN_DEFINITION_RENDER_KEY => $layoutDefinition,
      '#' . PintoLayoutHooks::PLUGIN_CONFIGURATION_RENDER_KEY => PluginConfiguration::fromData($this->getConfiguration()),
    ];

    // LB expects a certain structure:
    // We cannot clear out region data (blocks) here as LayoutBuilder::buildAdministrativeSection needs to decorate each block with attributes. (`build[$region][$uuid]['#attributes']['class'][]` etc).
    foreach ($layoutDefinition->regions->regions as $regionName) {
      if (\array_key_exists($regionName, $regions)) {
        $build[$regionName] = $regions[$regionName];
      }
    }

    return $build;
  }

  private function getFrozenDefinition(): FrozenLayoutDefinition {
    $pintoEnum = $this->getPluginDefinition()->get(DrupalLayoutDefinitionRepository::LAYOUT_PLUGIN_ADDITIONAL_PINTO_ENUM);
    if (!$pintoEnum instanceof ObjectListInterface) {
      // As set by  DrupalLayoutDefinitionRepository::getDefinitionsAsHookLayouts.
      throw new \LogicException('Impossible! Layout definition missing Pinto object enum');
    }

    return $this->drupalLayoutDefinitionRepository->getFrozenDefinitionByEnum($pintoEnum);
  }

  /**
   * @phpstan-return class-string<\Drupal\Core\Plugin\PluginFormInterface>
   */
  public function getFormClass($operation) {
    if ($this->realHasFormClass($operation)) {
      // @phpstan-ignore-next-line offsetAccess.nonOffsetAccessible return.type
      return $this->getPluginDefinition()->get(DrupalLayoutDefinitionRepository::LAYOUT_PLUGIN_ADDITIONAL_FORMS)[$operation];
    }
    else {
      return NullPluginForm::class;
    }
  }

  public function hasFormClass($operation): bool {
    // No matter what, we must always return a form. If this class *can* return forms: then it must always return forms.
    return TRUE;
  }

  private function realHasFormClass(string $operation): bool {
    /** @var array<string, class-string<\Drupal\Core\Plugin\PluginFormInterface>> $formClasses */
    $formClasses = $this->getPluginDefinition()->get(DrupalLayoutDefinitionRepository::LAYOUT_PLUGIN_ADDITIONAL_FORMS);
    return \array_key_exists($operation, $formClasses);
  }

}
