<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Render\Element;
use Drupal\pinto_layout\Discovery\RegionDataMode;
use Drupal\pinto_layout\DrupalLayoutDiscovery\DrupalLayoutDefinitionRepository;
use Drupal\pinto_layout\PintoLayout\Data\LayoutData;
use Drupal\pinto_layout\PintoLayout\Data\RegionAttributes;
use Drupal\pinto_layout\PintoLayout\Data\RegionData;
use Pinto\Attribute\Definition;
use Pinto\List\ObjectListInterface;

/**
 * @internal
 */
final class PintoLayoutHooks {

  public const FAUX_THEME_ELEMENT = 'pinto_layout_render';
  public const FROZEN_DEFINITION_RENDER_KEY = 'pinto_layout_frozen_definition';
  public const PLUGIN_CONFIGURATION_RENDER_KEY = 'pinto_layout_plugin_configuration';
  public const FAUX_RENDER_ELEMENT = 'pinto_layout_render_element';
  public const FAUX_RENDER_ELEMENT_PINTO_OBJECT = 'pinto_layout_render_pinto_object';

  public function __construct(
    private readonly DrupalLayoutDefinitionRepository $layoutDefinitionRepository,
  ) {
  }

  /**
   * Implements hook_theme().
   */
  #[Hook('theme')]
  public function theme(): array {
    $hooks = [];

    // Faux element
    // A fake element collects all the data that layout_discovery/layout_builder applies to the element, then takes it
    // and allows a Pinto object to act on it all. Otherwise, we would render a Pinto object in a layout plugin,
    // then we'd pass the [variable based] render array and hope and pray that LB doesn't clobber it; or that Pinto
    // module doesn't change the render array structure.
    $hooks[static::FAUX_THEME_ELEMENT] = [
      // LayoutBuilder::buildAdministrativeSection expects a 'render element' type.
      'render element' => \Drupal\pinto_layout\Hook\PintoLayoutHooks::FAUX_RENDER_ELEMENT,
    ];

    return $hooks;
  }

  /**
   * Implements hook_layout_alter().
   *
   * @phpstan-param \Drupal\Core\Layout\LayoutDefinition[] $definitions
   * @see \Drupal\Core\Layout\LayoutPluginManager::__construct
   */
  #[Hook('layout_alter')]
  public function layoutAlter(&$definitions): void {
    $definitions += $this->layoutDefinitionRepository->getDefinitionsAsHookLayouts();
  }

  /**
   * @see template_preprocess_layout()
   * @internal
   */
  public function hookPreprocess(array &$variables, string $hook): void {
    if ($hook === static::FAUX_THEME_ELEMENT) {
      // Validates that the factory returns the same class as expected.
      /** @var array{'pinto_layout_render_element': (array<string, array<string, mixed>>) } $variables */
      $validate = static function (\ReflectionMethod $factoryMethod, ObjectListInterface $pintoEnum, mixed $actualBuiltObject): void {
        if (!\is_object($actualBuiltObject)) {
          throw new \LogicException('Expected an object.');
        }

        $rEnum = new \ReflectionClassConstant($pintoEnum::class, $pintoEnum->name);
        /** @var array<\ReflectionAttribute<\Pinto\Attribute\Definition>> $attributes */
        $attributes = $rEnum->getAttributes(Definition::class);
        $definition = ($attributes[0] ?? NULL)?->newInstance() ?? throw new \LogicException('Missing definition.');

        if ($actualBuiltObject::class !== $definition->className) {
          throw new \Exception(\sprintf('Expected factory `%s` to return a `%s` but got a `%s`', (\sprintf('%s::%s', $factoryMethod->getDeclaringClass()->getName(), $factoryMethod->getName())), $definition->className, $actualBuiltObject::class));
        }
      };

      // @todo move all this into a class of its own

      /** @var \Drupal\pinto_layout\Discovery\FrozenLayoutDefinition $frozenDefinition */
      // @phpstan-ignore-next-line varTag.type
      $frozenDefinition = $variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT][\sprintf('#%s', PintoLayoutHooks::FROZEN_DEFINITION_RENDER_KEY)];
      /** @var \Drupal\pinto_layout\PintoLayout\Data\PluginConfiguration $pluginConfiguration */
      // @phpstan-ignore-next-line varTag.type
      $pluginConfiguration = $variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT][\sprintf('#%s', PintoLayoutHooks::PLUGIN_CONFIGURATION_RENDER_KEY)];

      $regionData = RegionData::fromData(regionsData: \array_fill_keys($frozenDefinition->regions->regions, []));
      foreach ($frozenDefinition->regions->regions as $regionName) {
        $originalRegionData = $variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT][$regionName] ?? [];
        foreach (Element::children($originalRegionData) as $childKey) {
          $regionData->add($regionName, $childKey, $originalRegionData[$childKey]);
        }
      }

      $regionAttributes = RegionAttributes::fromVariables($frozenDefinition, $variables);

      $methodArguments = match ($frozenDefinition->regionDataMode) {
        RegionDataMode::AsLayoutDataSingleParameter => ['layoutData' => LayoutData::fromRegionData($regionData, $regionAttributes, $pluginConfiguration)],
        // For AsParameterNamesFromRegionNames, we already validated during discovery that the only arguments that need population are regions.
        // Everything else must have a parameter default value.
        RegionDataMode::AsParameterNamesFromRegionNames => $regionData->regionsData,
      };

      foreach ($frozenDefinition->layoutDataAsParameter as $parameterName) {
        $methodArguments[$parameterName] = LayoutData::fromRegionData($regionData, $regionAttributes, $pluginConfiguration);
      }

      foreach ($frozenDefinition->regionAttributesAsParameter as $regionAttributeParameterName) {
        $methodArguments[$regionAttributeParameterName] = $regionAttributes;
      }

      foreach ($frozenDefinition->frozenLayoutDefinitionAsParameter as $parameterName) {
        $methodArguments[$parameterName] = $frozenDefinition;
      }

      foreach ($frozenDefinition->pluginConfigurationAsParameter as $parameterName) {
        $methodArguments[$parameterName] = $pluginConfiguration;
      }

      $rFactoryMethod = \ReflectionMethod::createFromMethodName($frozenDefinition->factoryMethod);

      // Remove arguments which are not parameters (this object probably uses LayoutData).
      $actualParamNames = \array_flip(\array_map(static fn(\ReflectionParameter $rParam): string => $rParam->getName(), $rFactoryMethod->getParameters()));
      $methodArguments = \array_intersect_key($methodArguments, $actualParamNames);

      if ($rFactoryMethod->isConstructor()) {
        $obj = $rFactoryMethod->getDeclaringClass()->newInstance(...$methodArguments);
      }
      elseif ($rFactoryMethod->isStatic()) {
        $obj = $rFactoryMethod->invoke(NULL, ...$methodArguments);
      }
      else {
        throw new \LogicException('Unhandled!');
      }

      $validate($rFactoryMethod, $frozenDefinition->pintoEnum, $obj);

      // @todo need to handle non callables!
      // @phpstan-ignore-next-line callable.nonCallable
      $build = $obj();

      $variables[PintoLayoutHooks::FAUX_RENDER_ELEMENT_PINTO_OBJECT] = $build;

      // Sadly, cannot mutate $build into $variables (via $variables = $build;), thusly changing $variables[#theme], directly as Renderer already
      // decided on the twig template.
      // Redefine everything, removing things like various attributes and prefixes.
      $variables = \array_intersect_key($variables, \array_flip([
        'directory',
        'theme_hook_original',
        PintoLayoutHooks::FAUX_RENDER_ELEMENT_PINTO_OBJECT,
      ]));
    }
  }

}
