<?php

declare(strict_types=1);

namespace Drupal\pinto_layout\PintoLayout\External;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\pinto_layout\Attribute\Regions;
use Drupal\pinto_layout\Discovery\FrozenLayoutDefinition;
use Drupal\pinto_layout\Discovery\PintoLayoutDiscovery;
use Pinto\List\ObjectListInterface;

final class ExternallyDefined {

  /**
   * @phpstan-param array<string, class-string<\Drupal\Core\Plugin\PluginFormInterface>> $forms
   */
  private function __construct(
    private string $id,
    private TranslatableMarkup $label,
    private readonly ObjectListInterface $pintoEnum,
    private readonly Regions $regions,
    private \ReflectionMethod $factoryMethod,
    public array $forms,
  ) {
  }

  /**
   * @phpstan-param Regions|list<string|\Drupal\pinto_layout\Attribute\Region> $regions
   * @phpstan-param callable|callable-string $factoryMethod
   * @phpstan-param array<string, class-string<\Drupal\Core\Plugin\PluginFormInterface>> $forms
   */
  public static function create(
    string $id,
    TranslatableMarkup $label,
    ObjectListInterface $pintoEnum,
    Regions|array $regions,
    callable|string $factoryMethod,
    array $forms = [],
  ): static {
    /** @var array|string|\Closure $factoryMethod */
    // @phpstan-ignore-next-line varTag.nativeType
    if ($factoryMethod instanceof \Closure) {
      throw new \Exception('$factoryMethod must be statically callable.');
    }

    $rMethod = \is_array($factoryMethod)
      ? new \ReflectionMethod($factoryMethod[0], $factoryMethod[1])
      : \ReflectionMethod::createFromMethodName($factoryMethod);

    return new static(
      id: $id,
      label: $label,
      pintoEnum: $pintoEnum,
      regions: \is_array($regions) ? new Regions(regions: $regions) : $regions,
      factoryMethod: $rMethod,
      forms: $forms,
    );
  }

  /**
   * @internal
   */
  public function freeze(): FrozenLayoutDefinition {
    $args = PintoLayoutDiscovery::determineFactoryMethod(
      $this->factoryMethod->getDeclaringClass(),
      $this->factoryMethod,
    );

    return new FrozenLayoutDefinition(
      $this->id,
      $this->label,
      $this->pintoEnum,
      $this->regions,
      ...[
        ...$args,
        ...['forms' => $this->forms],
      ],
    );
  }

}
