<?php

declare(strict_types=1);

namespace Drupal\pinto_theme\Plugin\DisplayVariant;

use Drupal\Core\Display\Attribute\PageDisplayVariant;
use Drupal\Core\Display\PageVariantInterface;
use Drupal\Core\Display\VariantBase;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\pinto_theme\Render\PintoRenderContext;
use Drupal\pinto_theme\Render\PintoThemeRenderer;

/**
 * Display variant for Pinto Theme.
 *
 * Configuration comes from
 * \Drupal\pinto_theme\EventListener\PintoThemePageDisplayVariant.
 *
 * @phpstan-type PintoThemePageVariantConfiguration array{
 *    pintoTheme: \Pinto\List\ObjectListInterface,
 *    pintoThemeContext: array{\Drupal\Core\Routing\RouteMatchInterface},
 *  }
 * @method PintoThemePageVariantConfiguration getConfiguration()
 */
#[PageDisplayVariant(
  id: self::PLUGIN_ID,
  admin_label: new TranslatableMarkup('Pinto Theme'),
)]
final class PintoThemePageVariant extends VariantBase implements PageVariantInterface {

  public const PLUGIN_ID = 'pinto_theme';

  protected array $mainContent = [];

  protected string|array|\Stringable $title = '';

  /**
   * {@inheritdoc}
   */
  public function setMainContent(array $main_content) {
    $this->mainContent = $main_content;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setTitle($title) {
    $this->title = $title;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    ['pintoTheme' => $htmlCase, 'pintoThemeContext' => $context] = $this->getConfiguration();
    [$routeMatch] = $context;

    // Abuse rendering.
    // Picked up by \Drupal\pinto_theme\Render\PintoThemeRenderer::render().
    return [
      PintoThemeRenderer::PINTO_THEME_CONTEXT => PintoRenderContext::createContextFrom(
        htmlEnum: $htmlCase,
        title: \is_array($this->title) ? $this->title : ['#markup' => $this->title],
        content: $this->mainContent,
        routeMatch: $routeMatch,
      ),
    ];
  }

}
