<?php

declare(strict_types=1);

namespace Drupal\pinto_theme\Html;

use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\AttachmentsInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Template\Attribute;

/**
 * Compiler pass to determine which Pinto lists represent a theme via #[Theme].
 *
 * Glue similar to \template_preprocess_html().
 *
 * @see \template_preprocess_html()
 * @see \Drupal\Core\Theme\ThemePreprocess::preprocessHtml
 */
final class DrupalHtmlTemplate {

  /**
   * @var non-empty-string
   */
  private string $placeHolderToken;

  private function __construct(
    private LanguageManagerInterface $languageManager,
    private RendererInterface $renderer,
  ) {
    // @phpstan-ignore-next-line
    $this->placeHolderToken = Crypt::randomBytesBase64(55);
  }

  public static function createFromContainer(ContainerInterface $container): static {
    return new static(
      $container->get('language_manager'),
      $container->get(RendererInterface::class),
    );
  }

  /**
   * @phpstan-return non-empty-string
   */
  public function placeHolderToken(): string {
    return $this->placeHolderToken;
  }

  public function htmlAttributes(): Attribute {
    $language = $this->languageManager->getCurrentLanguage();
    $htmlAttributes = new Attribute();
    $htmlAttributes['lang'] = $language->getId();
    $htmlAttributes['dir'] = $language->getDirection();
    return $htmlAttributes;
  }

  public function applyAttachments(AttachmentsInterface $to): void {
    // Create placeholder strings:
    $to->addAttachments([
      'html_response_attachment_placeholders' => \array_map(function (string $placeholderName): string {
        // Items are processed in \Drupal\Core\Render\HtmlResponseAttachmentsProcessor::processAttachments.
        return \sprintf('<%s-placeholder token="%s">', $placeholderName, $this->placeHolderToken);
      }, [
        // Keys are per \Drupal\Core\Render\HtmlResponseAttachmentsProcessor::process*(), values are per template, but
        // get their `-placeholder` suffix in the callback.
        'styles' => 'css',
        'scripts' => 'js',
        'scripts_bottom' => 'js-bottom',
        'head' => 'head',
      ]),
    ]);
  }

  /**
   * @phpstan-param array{'#markup': \Drupal\Component\Render\MarkupInterface|string} $title
   */
  public function renderTitle(array $title): string {
    return (string) $this->renderer->render($title);
  }

}
