<?php

declare(strict_types=1);

namespace Drupal\pinto_theme;

use Drupal\pinto\PintoMappingFactory;
use Drupal\pinto_theme\PintoTheme\Attribute\Theme;
use Drupal\pinto_theme\PintoTheme\Html\PintoThemeHtmlObjectInterface;
use Pinto\List\Resource\ObjectListEnumResource;
use Symfony\Component\DependencyInjection\Compiler\CompilerPassInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * Compiler pass to determine which Pinto lists represent a theme via #[Theme].
 */
final class PintoThemeCompilerPass implements CompilerPassInterface {

  public function process(ContainerBuilder $container): void {
    /** @var array<string, \Drupal\pinto_theme\PintoTheme\Html\PintoThemeHtmlObjectInterface> $mapping */
    $mapping = [];

    /** @var array<class-string, \Pinto\Resource\ResourceInterface> $resources */
    // @phpstan-ignore-next-line argument.type
    $resources = \unserialize($container->getDefinition(PintoMappingFactory::class)->getArgument('$resources'));

    foreach ($resources as $resource) {
      $componentClass = $resource->getClass();
      if (NULL === $componentClass) {
        continue;
      }

      $rClass = new \ReflectionClass($componentClass);

      $themes = [];
      foreach ($rClass->getAttributes(Theme::class) as $attr) {
        $themes[] = $attr->newInstance()->themeName;
        $mapping[$attr->newInstance()->themeName] = $componentClass;
      }

      if ($resource instanceof ObjectListEnumResource) {
        $case = $resource->pintoEnum;
        foreach ((new \ReflectionEnumUnitCase($case::class, $case->name))->getAttributes(Theme::class) as $attr) {
          $themes[] = $attr->newInstance()->themeName;
        }
      }

      if ($themes !== []) {
        if (FALSE === $rClass->implementsInterface(PintoThemeHtmlObjectInterface::class)) {
          throw new \Exception(\sprintf('Resources associated with a #[%s] attribute must implement %s', Theme::class, PintoThemeHtmlObjectInterface::class));
        }

        foreach ($themes as $theme) {
          $mapping[$theme] = $componentClass;
        }
      }
    }

    // Serialize until https://www.drupal.org/project/drupal/issues/3522410 is
    // resolved.
    $container->setParameter('pinto_theme.components', \serialize($mapping));
  }

}
